/**
 * Copyright (c) 2018, Ouster, Inc.
 * All rights reserved.
 */

#pragma once

#include <cstddef>

namespace ouster {
namespace viz {

inline float** genPalette(const int n, const float from[3], const float to[3]) {
    float** palette = new float*[n];
    for (int i = 0; i < n; i++) {
        palette[i] = new float[3];
        for (int j = 0; j < 3; j++) {
            palette[i][j] = (to[j] * i + from[j] * (n - 1 - i)) / (n - 1);
        }
    }
    return palette;
}

// clang-format off
// generated from:
// https://daniel.lawrence.lu/public/colortransform/#0_2423_964_352_6_2624_1000_513_11_3248_1000_617_15_415_1000_774
const size_t spezia_n = 256;
const float spezia_palette[spezia_n][3] = {
    {0.0489092208, 0.3426569998, 0.5139042139},
    {0.048956722, 0.3439922929, 0.5173324943},
    {0.0489996932, 0.3453243077, 0.5207851529},
    {0.0490380675, 0.3466530144, 0.5242624879},
    {0.0490717851, 0.3479782045, 0.5277650356},
    {0.0491007753, 0.3492998481, 0.5312931538},
    {0.0491249636, 0.3506177664, 0.5348473191},
    {0.0491442829, 0.3519318998, 0.5384278893},
    {0.0491586551, 0.3532420695, 0.5420354605},
    {0.0491680056, 0.3545481861, 0.5456704497},
    {0.0491722599, 0.3558500707, 0.5493333936},
    {0.0491713323, 0.357147634, 0.5530247092},
    {0.0491651408, 0.358440727, 0.5567449331},
    {0.0491536036, 0.3597291708, 0.5604946017},
    {0.0491366349, 0.3610128462, 0.5642743111},
    {0.0491141379, 0.3622916043, 0.5680845976},
    {0.0490860268, 0.3635652661, 0.5719259381},
    {0.0490522012, 0.364833653, 0.5757991076},
    {0.0490125641, 0.3660966158, 0.5797045231},
    {0.048967015, 0.3673539758, 0.5836429596},
    {0.0489154495, 0.3686055243, 0.5876150131},
    {0.0488577634, 0.3698511124, 0.5916212797},
    {0.0487938374, 0.3710905015, 0.5956625342},
    {0.0487235598, 0.372323513, 0.5997394919},
    {0.0486468151, 0.3735499382, 0.6038527489},
    {0.0485634804, 0.3747695088, 0.6080031991},
    {0.0484734215, 0.3759820759, 0.6121914983},
    {0.0483765155, 0.3771873116, 0.6164185405},
    {0.0482726209, 0.378385067, 0.6206851006},
    {0.0481615961, 0.3795750141, 0.6249919534},
    {0.0480432995, 0.3807569146, 0.6293400526},
    {0.0479175784, 0.3819304705, 0.6337302923},
    {0.0477842726, 0.3830954432, 0.6381635666},
    {0.0476432219, 0.3842515051, 0.6426408887},
    {0.0474942587, 0.3853983283, 0.6471631527},
    {0.0473372005, 0.3865356445, 0.6517314911},
    {0.0471718684, 0.3876630366, 0.6563468575},
    {0.0469980724, 0.3887802362, 0.6610103846},
    {0.0468156151, 0.3898868561, 0.6657232046},
    {0.0466242842, 0.3909824789, 0.6704865098},
    {0.0464238673, 0.392066747, 0.6753014326},
    {0.046214141, 0.3931392431, 0.6801693439},
    {0.0459948666, 0.3941994905, 0.6850914359},
    {0.0457657985, 0.3952471018, 0.6900690198},
    {0.0455266871, 0.3962815702, 0.6951036453},
    {0.0452772528, 0.3973024189, 0.7001965642},
    {0.0450172238, 0.3983091116, 0.7053493857},
    {0.0447462983, 0.3993011415, 0.7105636597},
    {0.0444641709, 0.4002778828, 0.7158409357},
    {0.04417051, 0.4012387991, 0.7211829424},
    {0.0438649841, 0.4021832347, 0.7265913486},
    {0.0435472243, 0.4031105638, 0.7320680022},
    {0.0432168543, 0.404020071, 0.7376147509},
    {0.0428734757, 0.4049110413, 0.7432335615},
    {0.0425166674, 0.4057827592, 0.7489264607},
    {0.0421459824, 0.4066343606, 0.7546955347},
    {0.0417609513, 0.4074650705, 0.7605429888},
    {0.0413610749, 0.4082739353, 0.7664710879},
    {0.0409458242, 0.4090600908, 0.772482276},
    {0.0405146405, 0.4098225236, 0.7785789967},
    {0.0400615036, 0.4105602205, 0.784763813},
    {0.0395929404, 0.4112720191, 0.7910394669},
    {0.0391097926, 0.4119568467, 0.7974087596},
    {0.0386117212, 0.4126133919, 0.8038747311},
    {0.0380983725, 0.4132404327, 0.8104403615},
    {0.0375693813, 0.4138365388, 0.8171089292},
    {0.0370243564, 0.4144002497, 0.8238838315},
    {0.0364628993, 0.4149300456, 0.8307685852},
    {0.0358845927, 0.4154242277, 0.8377668858},
    {0.0352889895, 0.4158810973, 0.8448827267},
    {0.0346756317, 0.4162987173, 0.8521200418},
    {0.0340440311, 0.4166751504, 0.8594833016},
    {0.033393681, 0.4170082211, 0.8669768572},
    {0.0327240452, 0.4172956645, 0.8746055365},
    {0.032034561, 0.4175350666, 0.8823743463},
    {0.031324625, 0.4177237749, 0.8902885318},
    {0.0305936169, 0.417858988, 0.8983536363},
    {0.029840868, 0.4179377258, 0.9065755606},
    {0.029065676, 0.4179567397, 0.9149603844},
    {0.028267296, 0.417912513, 0.9235146642},
    {0.0274449401, 0.4178012908, 0.9322453737},
    {0.0265977662, 0.4176190495, 0.9411597848},
    {0.0257248823, 0.4173613191, 0.950265646},
    {0.0248253364, 0.4170233607, 0.9595712423},
    {0.0238981117, 0.4165999591, 0.9690852165},
    {0.0229421277, 0.4160855114, 0.9788169265},
    {0.0219562203, 0.4154738188, 0.988776207},
    {0.035335727, 0.4150344729, 0.9966419339},
    {0.0820674896, 0.415476054, 0.9968754649},
    {0.113166444, 0.4159292281, 0.9971067309},
    {0.1377759725, 0.4163940251, 0.997335732},
    {0.1586260945, 0.4168703556, 0.9975625873},
    {0.1769588143, 0.4173582792, 0.9977872968},
    {0.1934603006, 0.4178577662, 0.9980098009},
    {0.2085556835, 0.4183687866, 0.9982302189},
    {0.2225293815, 0.4188913405, 0.9984484911},
    {0.2355824113, 0.4194253981, 0.9986646771},
    {0.2478629053, 0.4199709594, 0.998878777},
    {0.2594836354, 0.4205280244, 0.9990908504},
    {0.2705327868, 0.4210965335, 0.9993008971},
    {0.281080693, 0.4216764867, 0.9995089173},
    {0.2911846638, 0.422267884, 0.999714911},
    {0.3008919358, 0.4228706658, 0.9999189377},
    {0.3199598491, 0.4211529493, 1.0},
    {0.3436114788, 0.417874217, 1.0},
    {0.3653967679, 0.414583087, 1.0},
    {0.3856661618, 0.4112777412, 1.0},
    {0.4046753049, 0.4079563618, 1.0},
    {0.4226172864, 0.4046169817, 1.0},
    {0.4396421909, 0.4012576044, 1.0},
    {0.4558693767, 0.3978761435, 1.0},
    {0.4713956714, 0.3944704235, 1.0},
    {0.4863007963, 0.3910381794, 1.0},
    {0.5006514788, 0.3875770271, 1.0},
    {0.5145041347, 0.3840844929, 1.0},
    {0.5279071331, 0.3805579841, 1.0},
    {0.5409020782, 0.3769947588, 1.0},
    {0.5535253882, 0.3733918965, 1.0},
    {0.5658089519, 0.369746387, 1.0},
    {0.577780962, 0.3660549819, 1.0},
    {0.5894665718, 0.3623142838, 1.0},
    {0.6008882523, 0.3585206568, 1.0},
    {0.6120662093, 0.3546702862, 1.0},
    {0.6230187416, 0.350759089, 1.0},
    {0.6337625384, 0.3467826247, 1.0},
    {0.644312799, 0.342736274, 1.0},
    {0.6546834707, 0.3386149704, 1.0},
    {0.6648874283, 0.3344133496, 1.0},
    {0.674936533, 0.3301255703, 1.0},
    {0.6848417521, 0.3257453442, 1.0},
    {0.6946134567, 0.3212659061, 1.0},
    {0.7042612433, 0.3166798055, 1.0},
    {0.7137940526, 0.3119790256, 1.0},
    {0.7232204676, 0.3071546853, 1.0},
    {0.7325484753, 0.3021971881, 1.0},
    {0.7417856455, 0.2970957756, 1.0},
    {0.75093925, 0.2918387353, 1.0},
    {0.7600160837, 0.2864129543, 1.0},
    {0.769022882, 0.2808037996, 1.0},
    {0.7779658437, 0.27499488, 1.0},
    {0.7868511677, 0.2689677179, 1.0},
    {0.7956846356, 0.262701273, 1.0},
    {0.8044720292, 0.2561715841, 1.0},
    {0.813218832, 0.249350965, 1.0},
    {0.8219305873, 0.2422073185, 1.0},
    {0.830612421, 0.2347029448, 1.0},
    {0.8392695785, 0.2267931402, 1.0},
    {0.8479072452, 0.2184241861, 1.0},
    {0.8565303087, 0.2095306069, 1.0},
    {0.8651439548, 0.2000311613, 1.0},
    {0.8737530112, 0.1898229718, 1.0},
    {0.8823624253, 0.1787724197, 1.0},
    {0.8909772038, 0.1667005271, 1.0},
    {0.8996022344, 0.1533579528, 1.0},
    {0.9082425237, 0.1383788288, 1.0},
    {0.9169030786, 0.1211866736, 1.0},
    {0.925589025, 0.1007730514, 1.0},
    {0.9343054295, 0.0750433505, 1.0},
    {0.9430575371, 0.0378195234, 1.0},
    {0.9509350657, 0.0, 0.9989984632},
    {0.95544976, 0.0, 0.9943640232},
    {0.9599176645, 0.0, 0.9897799492},
    {0.9643412232, 0.0, 0.9852425456},
    {0.9687227607, 0.0, 0.9807481766},
    {0.9730644822, 0.0, 0.9762935042},
    {0.9773683548, 0.0, 0.9718751311},
    {0.981636405, 0.0, 0.9674898982},
    {0.9858704209, 0.0, 0.9631346464},
    {0.9900721908, 0.0, 0.9588062763},
    {0.9942433238, 0.0, 0.9545018077},
    {0.9983854294, 0.0, 0.9502183199},
    {1.0, 0.0280442368, 0.9437140822},
    {1.0, 0.0675265566, 0.9359017611},
    {1.0, 0.0944757834, 0.9282451868},
    {1.0, 0.1156788021, 0.920737803},
    {1.0, 0.1335218996, 0.9133734703},
    {1.0, 0.1491028368, 0.9061463475},
    {1.0, 0.1630325913, 0.8990508914},
    {1.0, 0.175691992, 0.8920819163},
    {1.0, 0.187337026, 0.8852344155},
    {1.0, 0.198149398, 0.8785036802},
    {1.0, 0.2082635462, 0.8718851209},
    {1.0, 0.2177821398, 0.8653745055},
    {1.0, 0.2267856747, 0.8589676619},
    {1.0, 0.2353385836, 0.8526607752},
    {1.0, 0.2434934378, 0.8464500904},
    {1.0, 0.2512937188, 0.8403319716},
    {1.0, 0.2587758601, 0.8343030214},
    {1.0, 0.2673909962, 0.8275538683},
    {1.0, 0.2793555558, 0.818752408},
    {1.0, 0.2906753719, 0.8101540804},
    {1.0, 0.3014349043, 0.8017491102},
    {1.0, 0.3117025793, 0.7935283184},
    {1.0, 0.3215346932, 0.7854831219},
    {1.0, 0.3309783041, 0.7776054144},
    {1.0, 0.3400730193, 0.7698873878},
    {1.0, 0.348852694, 0.762321949},
    {1.0, 0.3573464155, 0.7549021244},
    {1.0, 0.3655793965, 0.747621417},
    {1.0, 0.3735736609, 0.740473628},
    {1.0, 0.3813485801, 0.7334527969},
    {1.0, 0.3889212906, 0.7265533805},
    {1.0, 0.396306932, 0.7197698355},
    {1.0, 0.4035191238, 0.7130970955},
    {1.0, 0.4105699956, 0.7065300345},
    {1.0, 0.4174705744, 0.7000639439},
    {1.0, 0.4242307246, 0.6936940551},
    {1.0, 0.4308594465, 0.6874159575},
    {1.0, 0.4373649061, 0.6812251806},
    {1.0, 0.4437546134, 0.6751174927},
    {1.0, 0.4500353932, 0.6690886617},
    {1.0, 0.4562135339, 0.6631345749},
    {1.0, 0.462294817, 0.6572511792},
    {1.0, 0.4682845473, 0.6514344811},
    {1.0, 0.4741877317, 0.6456805468},
    {1.0, 0.4800088704, 0.6399853826},
    {1.0, 0.4857522845, 0.634344995},
    {1.0, 0.4914219081, 0.6287555695},
    {1.0, 0.4970214665, 0.6232129931},
    {1.0, 0.5025543571, 0.6177132726},
    {1.0, 0.5080239177, 0.6122524142},
    {1.0, 0.5134330988, 0.6068261862},
    {1.0, 0.5187847614, 0.6014304757},
    {1.0, 0.524081707, 0.5960607529},
    {1.0, 0.5293263197, 0.5907127261},
    {1.0, 0.5345211625, 0.5853816867},
    {1.0, 0.5396683812, 0.5800628662},
    {1.0, 0.5447702408, 0.5747513175},
    {1.0, 0.5498287678, 0.5694417953},
    {1.0, 0.5548459291, 0.5641288757},
    {1.0, 0.559823513, 0.558806777},
    {1.0, 0.5647634268, 0.5534694791},
    {1.0, 0.5696673393, 0.5481105447},
    {1.0, 0.57453686, 0.5427231789},
    {1.0, 0.5793736577, 0.5373001099},
    {1.0, 0.584179163, 0.5318335295},
    {1.0, 0.5889549851, 0.5263151526},
    {1.0, 0.5937024355, 0.5207359791},
    {1.0, 0.5984230042, 0.5150862932},
    {1.0, 0.603118062, 0.5093556046},
    {1.0, 0.6077888608, 0.5035324693},
    {1.0, 0.6124367714, 0.4976043701},
    {1.0, 0.6170630455, 0.4915577173},
    {1.0, 0.621669054, 0.4853774309},
    {1.0, 0.6262559295, 0.4790469706},
    {1.0, 0.6308249831, 0.4725479484},
    {1.0, 0.6353774667, 0.4658599198},
    {1.0, 0.6399146914, 0.4589600861},
    {1.0, 0.6444378495, 0.4518227875},
    {1.0, 0.6489482522, 0.444419086},
    {1.0, 0.6534472108, 0.4367160499},
    {1.0, 0.6579360366, 0.428676188},
    {1.0, 0.6624161005, 0.4202563167},
    {1.0, 0.6668888927, 0.411406368},
    {1.0, 0.6713557243, 0.4020678997},
    {1.0, 0.6758182049, 0.3921718895},
};


const size_t spezia_cal_ref_n = 256;
const float spezia_cal_ref_palette[spezia_cal_ref_n][3] = {
    {0.0489092208, 0.3426569998, 0.5139042139},
    {0.0490717851, 0.3479782045, 0.5277650356},
    {0.0491680056, 0.3545481861, 0.5456704497},
    {0.0491366349, 0.3610128462, 0.5642743111},
    {0.048967015, 0.3673539758, 0.5836429596},
    {0.0486468151, 0.3735499382, 0.6038527489},
    {0.0482726209, 0.378385067, 0.6206851006},
    {0.0476432219, 0.3842515051, 0.6426408887},
    {0.0469980724, 0.3887802362, 0.6610103846},
    {0.0459948666, 0.3941994905, 0.6850914359},
    {0.0450172238, 0.3983091116, 0.7053493857},
    {0.0438649841, 0.4021832347, 0.7265913486},
    {0.0425166674, 0.4057827592, 0.7489264607},
    {0.0409458242, 0.4090600908, 0.772482276},
    {0.0391097926, 0.4119568467, 0.7974087596},
    {0.0370243564, 0.4144002497, 0.8238838315},
    {0.0346756317, 0.4162987173, 0.8521200418},
    {0.032034561, 0.4175350666, 0.8823743463},
    {0.029065676, 0.4179567397, 0.9149603844},
    {0.0257248823, 0.4173613191, 0.950265646},
    {0.0229421277, 0.4160855114, 0.9788169265},
    {0.113166444, 0.4159292281, 0.9971067309},
    {0.1769588143, 0.4173582792, 0.9977872968},
    {0.2355824113, 0.4194253981, 0.9986646771},
    {0.2705327868, 0.4210965335, 0.9993008971},
    {0.3199598491, 0.4211529493, 1.0},
    {0.3856661618, 0.4112777412, 1.0},
    {0.4396421909, 0.4012576044, 1.0},
    {0.4863007963, 0.3910381794, 1.0},
    {0.5409020782, 0.3769947588, 1.0},
    {0.577780962, 0.3660549819, 1.0},
    {0.6120662093, 0.3546702862, 1.0},
    {0.644312799, 0.342736274, 1.0},
    {0.6648874283, 0.3344133496, 1.0},
    {0.6946134567, 0.3212659061, 1.0},
    {0.7232204676, 0.3071546853, 1.0},
    {0.75093925, 0.2918387353, 1.0},
    {0.7779658437, 0.27499488, 1.0},
    {0.7956846356, 0.262701273, 1.0},
    {0.8219305873, 0.2422073185, 1.0},
    {0.8392695785, 0.2267931402, 1.0},
    {0.8651439548, 0.2000311613, 1.0},
    {0.8823624253, 0.1787724197, 1.0},
    {0.9082425237, 0.1383788288, 1.0},
    {0.925589025, 0.1007730514, 1.0},
    {0.9509350657, 0.0, 0.9989984632},
    {0.9599176645, 0.0, 0.9897799492},
    {0.9687227607, 0.0, 0.9807481766},
    {0.9773683548, 0.0, 0.9718751311},
    {0.9858704209, 0.0, 0.9631346464},
    {0.9983854294, 0.0, 0.9502183199},
    {1.0, 0.0675265566, 0.9359017611},
    {1.0, 0.1156788021, 0.920737803},
    {1.0, 0.1491028368, 0.9061463475},
    {1.0, 0.175691992, 0.8920819163},
    {1.0, 0.198149398, 0.8785036802},
    {1.0, 0.2082635462, 0.8718851209},
    {1.0, 0.2267856747, 0.8589676619},
    {1.0, 0.2434934378, 0.8464500904},
    {1.0, 0.2587758601, 0.8343030214},
    {1.0, 0.2793555558, 0.818752408},
    {1.0, 0.2906753719, 0.8101540804},
    {1.0, 0.3117025793, 0.7935283184},
    {1.0, 0.3309783041, 0.7776054144},
    {1.0, 0.3400730193, 0.7698873878},
    {1.0, 0.3573464155, 0.7549021244},
    {1.0, 0.3655793965, 0.747621417},
    {1.0, 0.3813485801, 0.7334527969},
    {1.0, 0.3889212906, 0.7265533805},
    {1.0, 0.4035191238, 0.7130970955},
    {1.0, 0.4105699956, 0.7065300345},
    {1.0, 0.4174705744, 0.7000639439},
    {1.0, 0.4308594465, 0.6874159575},
    {1.0, 0.4373649061, 0.6812251806},
    {1.0, 0.4437546134, 0.6751174927},
    {1.0, 0.4562135339, 0.6631345749},
    {1.0, 0.462294817, 0.6572511792},
    {1.0, 0.4682845473, 0.6514344811},
    {1.0, 0.4741877317, 0.6456805468},
    {1.0, 0.4800088704, 0.6399853826},
    {1.0, 0.4914219081, 0.6287555695},
    {1.0, 0.4970214665, 0.6232129931},
    {1.0, 0.5025543571, 0.6177132726},
    {1.0, 0.5080239177, 0.6122524142},
    {1.0, 0.5134330988, 0.6068261862},
    {1.0, 0.5187847614, 0.6014304757},
    {1.0, 0.524081707, 0.5960607529},
    {1.0, 0.5293263197, 0.5907127261},
    {1.0, 0.5345211625, 0.5853816867},
    {1.0, 0.5396683812, 0.5800628662},
    {1.0, 0.5396683812, 0.5800628662},
    {1.0, 0.5447702408, 0.5747513175},
    {1.0, 0.5498287678, 0.5694417953},
    {1.0, 0.5548459291, 0.5641288757},
    {1.0, 0.559823513, 0.558806777},
    {1.0, 0.5647634268, 0.5534694791},
    {1.0, 0.5647634268, 0.5534694791},
    {1.0, 0.5696673393, 0.5481105447},
    {1.0, 0.57453686, 0.5427231789},
    {1.0, 0.5793736577, 0.5373001099},
    {1.0, 0.5793736577, 0.5373001099},
    {1.0, 0.584179163, 0.5318335295},
    {1.0, 0.5889549851, 0.5263151526},
    {1.0, 0.5889549851, 0.5263151526},
    {1.0, 0.5937024355, 0.5207359791},
    {1.0, 0.5937024355, 0.5207359791},
    {1.0, 0.5984230042, 0.5150862932},
    {1.0, 0.603118062, 0.5093556046},
    {1.0, 0.603118062, 0.5093556046},
    {1.0, 0.6077888608, 0.5035324693},
    {1.0, 0.6077888608, 0.5035324693},
    {1.0, 0.6124367714, 0.4976043701},
    {1.0, 0.6124367714, 0.4976043701},
    {1.0, 0.6170630455, 0.4915577173},
    {1.0, 0.6170630455, 0.4915577173},
    {1.0, 0.621669054, 0.4853774309},
    {1.0, 0.621669054, 0.4853774309},
    {1.0, 0.6262559295, 0.4790469706},
    {1.0, 0.6262559295, 0.4790469706},
    {1.0, 0.6262559295, 0.4790469706},
    {1.0, 0.6308249831, 0.4725479484},
    {1.0, 0.6308249831, 0.4725479484},
    {1.0, 0.6353774667, 0.4658599198},
    {1.0, 0.6353774667, 0.4658599198},
    {1.0, 0.6353774667, 0.4658599198},
    {1.0, 0.6399146914, 0.4589600861},
    {1.0, 0.6399146914, 0.4589600861},
    {1.0, 0.6399146914, 0.4589600861},
    {1.0, 0.6444378495, 0.4518227875},
    {1.0, 0.6444378495, 0.4518227875},
    {1.0, 0.6444378495, 0.4518227875},
    {1.0, 0.6489482522, 0.444419086},
    {1.0, 0.6489482522, 0.444419086},
    {1.0, 0.6489482522, 0.444419086},
    {1.0, 0.6489482522, 0.444419086},
    {1.0, 0.6534472108, 0.4367160499},
    {1.0, 0.6534472108, 0.4367160499},
    {1.0, 0.6534472108, 0.4367160499},
    {1.0, 0.6534472108, 0.4367160499},
    {1.0, 0.6579360366, 0.428676188},
    {1.0, 0.6579360366, 0.428676188},
    {1.0, 0.6579360366, 0.428676188},
    {1.0, 0.6579360366, 0.428676188},
    {1.0, 0.6579360366, 0.428676188},
    {1.0, 0.6579360366, 0.428676188},
    {1.0, 0.6624161005, 0.4202563167},
    {1.0, 0.6624161005, 0.4202563167},
    {1.0, 0.6624161005, 0.4202563167},
    {1.0, 0.6624161005, 0.4202563167},
    {1.0, 0.6624161005, 0.4202563167},
    {1.0, 0.6624161005, 0.4202563167},
    {1.0, 0.6668888927, 0.411406368},
    {1.0, 0.6668888927, 0.411406368},
    {1.0, 0.6668888927, 0.411406368},
    {1.0, 0.6668888927, 0.411406368},
    {1.0, 0.6668888927, 0.411406368},
    {1.0, 0.6668888927, 0.411406368},
    {1.0, 0.6668888927, 0.411406368},
    {1.0, 0.6668888927, 0.411406368},
    {1.0, 0.6713557243, 0.4020678997},
    {1.0, 0.6713557243, 0.4020678997},
    {1.0, 0.6713557243, 0.4020678997},
    {1.0, 0.6713557243, 0.4020678997},
    {1.0, 0.6713557243, 0.4020678997},
    {1.0, 0.6713557243, 0.4020678997},
    {1.0, 0.6713557243, 0.4020678997},
    {1.0, 0.6713557243, 0.4020678997},
    {1.0, 0.6713557243, 0.4020678997},
    {1.0, 0.6713557243, 0.4020678997},
    {1.0, 0.6713557243, 0.4020678997},
    {1.0, 0.6713557243, 0.4020678997},
    {1.0, 0.6758182049, 0.3921718895},
    {1.0, 0.6758182049, 0.3921718895},
    {1.0, 0.6758182049, 0.3921718895},
    {1.0, 0.6758182049, 0.3921718895},
    {1.0, 0.6758182049, 0.3921718895},
    {1.0, 0.6758182049, 0.3921718895},
    {1.0, 0.6758182049, 0.3921718895},
    {1.0, 0.6758182049, 0.3921718895},
    {1.0, 0.6758182049, 0.3921718895},
    {1.0, 0.6758182049, 0.3921718895},
    {1.0, 0.6758182049, 0.3921718895},
    {1.0, 0.6758182049, 0.3921718895},
    {1.0, 0.6758182049, 0.3921718895},
    {1.0, 0.6758182049, 0.3921718895},
    {1.0, 0.6758182049, 0.3921718895},
    {1.0, 0.6758182049, 0.3921718895},
    {1.0, 0.6758182049, 0.3921718895},
    {1.0, 0.6758182049, 0.3921718895},
    {1.0, 0.6758182049, 0.3921718895},
    {1.0, 0.6758182049, 0.3921718895},
    {1.0, 0.6758182049, 0.3921718895},
    {1.0, 0.6758182049, 0.3921718895},
    {1.0, 0.6758182049, 0.3921718895},
    {1.0, 0.6758182049, 0.3921718895},
    {1.0, 0.6758182049, 0.3921718895},
    {1.0, 0.6758182049, 0.3921718895},
    {1.0, 0.6758182049, 0.3921718895},
    {1.0, 0.6758182049, 0.3921718895},
    {1.0, 0.6758182049, 0.3921718895},
    {1.0, 0.6758182049, 0.3921718895},
    {1.0, 0.6758182049, 0.3921718895},
    {1.0, 0.6758182049, 0.3921718895},
    {1.0, 0.6758182049, 0.3921718895},
    {1.0, 0.6758182049, 0.3921718895},
    {1.0, 0.6758182049, 0.3921718895},
    {1.0, 0.6758182049, 0.3921718895},
    {1.0, 0.6758182049, 0.3921718895},
    {1.0, 0.6758182049, 0.3921718895},
    {1.0, 0.6758182049, 0.3921718895},
    {1.0, 0.6758182049, 0.3921718895},
    {1.0, 0.6758182049, 0.3921718895},
    {1.0, 0.6758182049, 0.3921718895},
    {1.0, 0.6758182049, 0.3921718895},
    {1.0, 0.6758182049, 0.3921718895},
    {1.0, 0.6758182049, 0.3921718895},
    {1.0, 0.6758182049, 0.3921718895},
    {1.0, 0.6758182049, 0.3921718895},
    {1.0, 0.6758182049, 0.3921718895},
    {1.0, 0.6758182049, 0.3921718895},
    {1.0, 0.6758182049, 0.3921718895},
    {1.0, 0.6758182049, 0.3921718895},
    {1.0, 0.6758182049, 0.3921718895},
    {1.0, 0.6758182049, 0.3921718895},
    {1.0, 0.6758182049, 0.3921718895},
    {1.0, 0.6758182049, 0.3921718895},
    {1.0, 0.6758182049, 0.3921718895},
    {1.0, 0.6758182049, 0.3921718895},
    {1.0, 0.6758182049, 0.3921718895},
    {1.0, 0.6758182049, 0.3921718895},
    {1.0, 0.6758182049, 0.3921718895},
    {1.0, 0.6758182049, 0.3921718895},
    {1.0, 0.6758182049, 0.3921718895},
    {1.0, 0.6758182049, 0.3921718895},
    {1.0, 0.6758182049, 0.3921718895},
    {1.0, 0.6758182049, 0.3921718895},
    {1.0, 0.6758182049, 0.3921718895},
    {1.0, 0.6758182049, 0.3921718895},
    {1.0, 0.6758182049, 0.3921718895},
    {1.0, 0.6758182049, 0.3921718895},
    {1.0, 0.6758182049, 0.3921718895},
    {1.0, 0.6758182049, 0.3921718895},
    {1.0, 0.6758182049, 0.3921718895},
    {1.0, 0.6758182049, 0.3921718895},
    {1.0, 0.6758182049, 0.3921718895},
    {1.0, 0.6758182049, 0.3921718895},
    {1.0, 0.6758182049, 0.3921718895},
    {1.0, 0.6758182049, 0.3921718895},
    {1.0, 0.6758182049, 0.3921718895},
    {1.0, 0.6758182049, 0.3921718895},
    {1.0, 0.6758182049, 0.3921718895},
    {1.0, 0.6758182049, 0.3921718895},
    {1.0, 0.6758182049, 0.3921718895},
    {1.0, 0.6758182049, 0.3921718895},
    {1.0, 0.6758182049, 0.3921718895},
    {1.0, 0.6758182049, 0.3921718895},
};

const size_t calref_n = 256;
const float calref_palette[calref_n][3] = {
    {0.37562745098039246f, 0.30080392156862746f, 0.6312941176470588f},
    {0.36862745098039246f, 0.30980392156862746f, 0.6352941176470588f},
    {0.3618608227604765f, 0.31856978085351784f, 0.6394463667820068f},
    {0.3550941945405613f, 0.3273356401384083f, 0.643598615916955f},
    {0.3483275663206459f, 0.3361014994232987f, 0.647750865051903f},
    {0.3415609381007305f, 0.3448673587081891f, 0.6519031141868512f},
    {0.33479430988081516f, 0.35363321799307956f, 0.6560553633217993f},
    {0.3280276816608997f, 0.36239907727796994f, 0.6602076124567474f},
    {0.3212610534409842f, 0.3711649365628603f, 0.6643598615916955f},
    {0.31449442522106885f, 0.3799307958477509f, 0.6685121107266436f},
    {0.3077277970011534f, 0.38869665513264134f, 0.6726643598615917f},
    {0.300961168781238f, 0.3974625144175317f, 0.6768166089965398f},
    {0.29419454056132255f, 0.4062283737024219f, 0.6809688581314879f},
    {0.2874279123414072f, 0.4149942329873126f, 0.685121107266436f},
    {0.2806612841214917f, 0.4237600922722031f, 0.6892733564013841f},
    {0.27389465590157624f, 0.4325259515570933f, 0.6934256055363321f},
    {0.2671280276816609f, 0.4412918108419839f, 0.6975778546712803f},
    {0.2603613994617455f, 0.45005767012687425f, 0.7017301038062282f},
    {0.25359477124183005f, 0.4588235294117643f, 0.7058823529411765f},
    {0.24682814302191458f, 0.46758938869665506f, 0.7100346020761246f},
    {0.24006151480199922f, 0.4763552479815456f, 0.7141868512110727f},
    {0.23329488658208386f, 0.485121107266436f, 0.7183391003460207f},
    {0.2265282583621684f, 0.49388696655132636f, 0.7224913494809689f},
    {0.21976163014225292f, 0.5026528258362168f, 0.726643598615917f},
    {0.21299500192233756f, 0.5114186851211073f, 0.7307958477508651f},
    {0.20622837370242209f, 0.5201845444059976f, 0.7349480968858132f},
    {0.19946174548250672f, 0.5289504036908883f, 0.7391003460207612f},
    {0.20007689350249913f, 0.5377931564782777f, 0.7393310265282583f},
    {0.2080738177623992f, 0.5467128027681663f, 0.7356401384083046f},
    {0.21607074202229903f, 0.5556324490580544f, 0.7319492502883508f},
    {0.2240676662821992f, 0.5645520953479432f, 0.7282583621683968f},
    {0.23206459054209927f, 0.5734717416378313f, 0.7245674740484429f},
    {0.24006151480199933f, 0.58239138792772f, 0.720876585928489f},
    {0.2480584390618994f, 0.5913110342176088f, 0.7171856978085351f},
    {0.25605536332179935f, 0.6002306805074966f, 0.7134948096885814f},
    {0.2640522875816994f, 0.6091503267973857f, 0.7098039215686275f},
    {0.27204921184159947f, 0.6180699730872741f, 0.7061130334486736f},
    {0.28004613610149953f, 0.6269896193771626f, 0.7024221453287197f},
    {0.2880430603613995f, 0.6359092656670511f, 0.6987312572087658f},
    {0.29603998462129966f, 0.6448289119569397f, 0.695040369088812f},
    {0.3040369088811996f, 0.6537485582468282f, 0.6913494809688581f},
    {0.3120338331410998f, 0.6626682045367166f, 0.6876585928489042f},
    {0.32003075740099973f, 0.671587850826605f, 0.6839677047289503f},
    {0.3280276816608998f, 0.6805074971164937f, 0.6802768166089965f},
    {0.33602460592079986f, 0.6894271434063821f, 0.6765859284890426f},
    {0.3440215301806999f, 0.6983467896962707f, 0.6728950403690888f},
    {0.35201845444059976f, 0.7072664359861591f, 0.6692041522491351f},
    {0.36001537870050004f, 0.7161860822760477f, 0.6655132641291811f},
    {0.3680123029604f, 0.7251057285659362f, 0.6618223760092272f},
    {0.37600922722029995f, 0.7340253748558248f, 0.6581314878892734f},
    {0.3840061514802f, 0.7429450211457131f, 0.6544405997693193f},
    {0.39200307574010007f, 0.7518646674356018f, 0.6507497116493657f},
    {0.40000000000000036f, 0.7607843137254902f, 0.6470588235294117f},
    {0.4106113033448675f, 0.7649365628604383f, 0.6469050365244137f},
    {0.42122260668973477f, 0.7690888119953864f, 0.6467512495194156f},
    {0.43183391003460214f, 0.7732410611303345f, 0.6465974625144175f},
    {0.4424452133794696f, 0.7773933102652826f, 0.6464436755094196f},
    {0.4530565167243371f, 0.7815455594002306f, 0.6462898885044215f},
    {0.46366782006920415f, 0.7856978085351789f, 0.6461361014994234f},
    {0.4742791234140715f, 0.7898500576701271f, 0.6459823144944252f},
    {0.4848904267589389f, 0.794002306805075f, 0.6458285274894271f},
    {0.49550173010380627f, 0.7981545559400232f, 0.645674740484429f},
    {0.5061130334486739f, 0.8023068050749711f, 0.6455209534794312f},
    {0.5167243367935411f, 0.8064590542099194f, 0.645367166474433f},
    {0.5273356401384084f, 0.8106113033448674f, 0.6452133794694349f},
    {0.5379469434832758f, 0.8147635524798154f, 0.6450595924644369f},
    {0.548558246828143f, 0.8189158016147636f, 0.6449058054594388f},
    {0.5591695501730105f, 0.8230680507497117f, 0.6447520184544406f},
    {0.5697808535178779f, 0.8272202998846598f, 0.6445982314494427f},
    {0.5803921568627453f, 0.831372549019608f, 0.6444444444444446f},
    {0.5910034602076126f, 0.8355247981545562f, 0.6442906574394465f},
    {0.60161476355248f, 0.8396770472895041f, 0.6441368704344483f},
    {0.6122260668973473f, 0.8438292964244521f, 0.6439830834294502f},
    {0.6228373702422147f, 0.8479815455594002f, 0.6438292964244523f},
    {0.633448673587082f, 0.8521337946943485f, 0.6436755094194541f},
    {0.6440599769319493f, 0.8562860438292964f, 0.6435217224144562f},
    {0.6546712802768165f, 0.8604382929642447f, 0.6433679354094579f},
    {0.6652825836216838f, 0.8645905420991928f, 0.6432141484044598f},
    {0.675124951941561f, 0.8685121107266438f, 0.6422145328719724f},
    {0.6841983852364476f, 0.8722029988465975f, 0.6403690888119954f},
    {0.6932718185313342f, 0.8758938869665513f, 0.6385236447520186f},
    {0.7023452518262208f, 0.8795847750865051f, 0.6366782006920415f},
    {0.7114186851211074f, 0.8832756632064591f, 0.6348327566320646f},
    {0.7204921184159938f, 0.8869665513264131f, 0.6329873125720877f},
    {0.7295655517108806f, 0.890657439446367f, 0.6311418685121105f},
    {0.7386389850057672f, 0.8943483275663208f, 0.6292964244521339f},
    {0.7477124183006536f, 0.8980392156862746f, 0.6274509803921569f},
    {0.7567858515955403f, 0.9017301038062284f, 0.62560553633218f},
    {0.7658592848904268f, 0.9054209919261822f, 0.6237600922722031f},
    {0.7749327181853134f, 0.909111880046136f, 0.6219146482122262f},
    {0.7840061514802001f, 0.9128027681660901f, 0.6200692041522492f},
    {0.7930795847750867f, 0.916493656286044f, 0.618223760092272f},
    {0.8021530180699734f, 0.920184544405998f, 0.6163783160322951f},
    {0.8112264513648599f, 0.9238754325259518f, 0.6145328719723183f},
    {0.8202998846597466f, 0.9275663206459055f, 0.6126874279123413f},
    {0.8293733179546331f, 0.9312572087658594f, 0.6108419838523645f},
    {0.8384467512495197f, 0.9349480968858133f, 0.6089965397923875f},
    {0.8475201845444063f, 0.9386389850057671f, 0.6071510957324106f},
    {0.8565936178392928f, 0.9423298731257211f, 0.6053056516724337f},
    {0.8656670511341793f, 0.9460207612456747f, 0.6034602076124568f},
    {0.874740484429066f, 0.9497116493656288f, 0.6016147635524798f},
    {0.8838139177239525f, 0.9534025374855826f, 0.5997693194925027f},
    {0.8928873510188393f, 0.9570934256055367f, 0.5979238754325257f},
    {0.9019607843137256f, 0.9607843137254903f, 0.5960784313725491f},
    {0.9058054594386773f, 0.962322183775471f, 0.6020761245674742f},
    {0.9096501345636295f, 0.9638600538254517f, 0.6080738177623993f},
    {0.9134948096885813f, 0.9653979238754326f, 0.6140715109573244f},
    {0.9173394848135333f, 0.9669357939254133f, 0.6200692041522493f},
    {0.9211841599384853f, 0.9684736639753941f, 0.6260668973471741f},
    {0.9250288350634372f, 0.9700115340253751f, 0.6320645905420991f},
    {0.9288735101883892f, 0.9715494040753557f, 0.6380622837370243f},
    {0.932718185313341f, 0.9730872741253366f, 0.6440599769319492f},
    {0.9365628604382931f, 0.9746251441753172f, 0.6500576701268744f},
    {0.9404075355632451f, 0.9761630142252982f, 0.6560553633217994f},
    {0.9442522106881969f, 0.9777008842752788f, 0.6620530565167244f},
    {0.9480968858131487f, 0.9792387543252595f, 0.6680507497116493f},
    {0.9519415609381008f, 0.9807766243752404f, 0.6740484429065746f},
    {0.9557862360630527f, 0.9823144944252212f, 0.6800461361014994f},
    {0.9596309111880046f, 0.9838523644752019f, 0.6860438292964245f},
    {0.9634755863129567f, 0.9853902345251826f, 0.6920415224913494f},
    {0.9673202614379086f, 0.9869281045751634f, 0.6980392156862747f},
    {0.9711649365628605f, 0.9884659746251442f, 0.7040369088811996f},
    {0.9750096116878124f, 0.9900038446751249f, 0.7100346020761246f},
    {0.9788542868127644f, 0.9915417147251058f, 0.7160322952710494f},
    {0.9826989619377164f, 0.9930795847750866f, 0.7220299884659747f},
    {0.9865436370626683f, 0.9946174548250674f, 0.7280276816608996f},
    {0.9903883121876201f, 0.9961553248750481f, 0.7340253748558248f},
    {0.9942329873125721f, 0.9976931949250287f, 0.7400230680507498f},
    {0.9980776624375239f, 0.9992310649750095f, 0.746020761245675f},
    {0.9999231064975008f, 0.9976163014225297f, 0.7450211457131873f},
    {0.9997693194925027f, 0.9928489042675892f, 0.7370242214532873f},
    {0.9996155324875048f, 0.988081507112649f, 0.729027297193387f},
    {0.9994617454825068f, 0.9833141099577085f, 0.7210303729334873f},
    {0.9993079584775085f, 0.9785467128027682f, 0.7130334486735873f},
    {0.9991541714725107f, 0.9737793156478278f, 0.7050365244136869f},
    {0.9990003844675125f, 0.9690119184928874f, 0.697039600153787f},
    {0.9988465974625144f, 0.9642445213379468f, 0.6890426758938869f},
    {0.9986928104575163f, 0.9594771241830067f, 0.681045751633987f},
    {0.9985390234525182f, 0.9547097270280661f, 0.6730488273740869f},
    {0.9983852364475202f, 0.9499423298731258f, 0.6650519031141869f},
    {0.9982314494425222f, 0.9451749327181854f, 0.6570549788542868f},
    {0.998077662437524f, 0.9404075355632449f, 0.6490580545943867f},
    {0.9979238754325258f, 0.9356401384083044f, 0.6410611303344868f},
    {0.9977700884275279f, 0.930872741253364f, 0.6330642060745867f},
    {0.9976163014225298f, 0.9261053440984237f, 0.6250672818146866f},
    {0.9974625144175316f, 0.9213379469434833f, 0.6170703575547868f},
    {0.9973087274125335f, 0.9165705497885427f, 0.6090734332948867f},
    {0.9971549404075356f, 0.9118031526336023f, 0.6010765090349864f},
    {0.9970011534025374f, 0.907035755478662f, 0.5930795847750865f},
    {0.9968473663975395f, 0.9022683583237218f, 0.5850826605151866f},
    {0.9966935793925413f, 0.8975009611687812f, 0.5770857362552864f},
    {0.9965397923875433f, 0.892733564013841f, 0.5690888119953864f},
    {0.9963860053825454f, 0.8879661668589005f, 0.5610918877354861f},
    {0.9962322183775473f, 0.88319876970396f, 0.5530949634755861f},
    {0.996078431372549f, 0.8784313725490196f, 0.5450980392156861f},
    {0.9959246443675508f, 0.8707420222991156f, 0.538638985005767f},
    {0.9957708573625528f, 0.8630526720492118f, 0.5321799307958477f},
    {0.9956170703575548f, 0.855363321799308f, 0.5257208765859284f},
    {0.9954632833525567f, 0.847673971549404f, 0.519261822376009f},
    {0.9953094963475586f, 0.8399846212995001f, 0.5128027681660898f},
    {0.9951557093425605f, 0.8322952710495963f, 0.5063437139561706f},
    {0.9950019223375625f, 0.8246059207996924f, 0.4998846597462513f},
    {0.9948481353325646f, 0.8169165705497885f, 0.4934256055363321f},
    {0.9946943483275664f, 0.8092272202998847f, 0.48696655132641264f},
    {0.9945405613225683f, 0.8015378700499808f, 0.48050749711649365f},
    {0.9943867743175702f, 0.7938485198000771f, 0.47404844290657466f},
    {0.9942329873125721f, 0.7861591695501731f, 0.4675893886966551f},
    {0.994079200307574f, 0.7784698193002692f, 0.4611303344867359f},
    {0.993925413302576f, 0.7707804690503652f, 0.4546712802768166f},
    {0.9937716262975778f, 0.7630911188004613f, 0.4482122260668975f},
    {0.99361783929258f, 0.7554017685505575f, 0.44175317185697793f},
    {0.9934640522875816f, 0.7477124183006536f, 0.43529411764705894f},
    {0.9933102652825835f, 0.7400230680507496f, 0.4288350634371395f},
    {0.9931564782775857f, 0.7323337178008458f, 0.4223760092272202f},
    {0.9930026912725872f, 0.724644367550942f, 0.4159169550173013f},
    {0.9928489042675894f, 0.716955017301038f, 0.40945790080738176f},
    {0.9926951172625912f, 0.7092656670511341f, 0.40299884659746266f},
    {0.9925413302575933f, 0.7015763168012303f, 0.3965397923875432f},
    {0.9923875432525952f, 0.6938869665513263f, 0.390080738177624f},
    {0.992233756247597f, 0.6861976163014225f, 0.3836216839677048f},
    {0.9914648212226067f, 0.677354863514033f, 0.3780853517877738f},
    {0.990080738177624f, 0.6673587081891583f, 0.3734717416378317f},
    {0.9886966551326414f, 0.6573625528642829f, 0.36885813148788904f},
    {0.9873125720876587f, 0.647366397539408f, 0.36424452133794694f},
    {0.985928489042676f, 0.6373702422145329f, 0.3596309111880045f},
    {0.9845444059976933f, 0.6273740868896578f, 0.35501730103806217f},
    {0.9831603229527106f, 0.6173779315647828f, 0.35040369088811985f},
    {0.981776239907728f, 0.6073817762399077f, 0.3457900807381774f},
    {0.9803921568627451f, 0.5973856209150328f, 0.3411764705882355f},
    {0.9790080738177624f, 0.5873894655901575f, 0.33656286043829287f},
    {0.9776239907727798f, 0.5773933102652827f, 0.33194925028835065f},
    {0.976239907727797f, 0.5673971549404075f, 0.3273356401384082f},
    {0.9748558246828143f, 0.5574009996155325f, 0.3227220299884661f},
    {0.9734717416378316f, 0.5474048442906574f, 0.3181084198385238f},
    {0.9720876585928488f, 0.5374086889657824f, 0.31349480968858146f},
    {0.9707035755478661f, 0.5274125336409075f, 0.30888119953863913f},
    {0.9693194925028835f, 0.5174163783160323f, 0.3042675893886967f},
    {0.9679354094579009f, 0.5074202229911575f, 0.2996539792387545f},
    {0.9665513264129182f, 0.4974240676662822f, 0.2950403690888119f},
    {0.9651672433679354f, 0.4874279123414072f, 0.2904267589388697f},
    {0.9637831603229527f, 0.47743175701653207f, 0.2858131487889273f},
    {0.9623990772779699f, 0.4674356016916571f, 0.28119953863898506f},
    {0.9610149942329872f, 0.4574394463667821f, 0.27658592848904273f},
    {0.9596309111880046f, 0.447443291041907f, 0.2719723183391005f},
    {0.958246828143022f, 0.43744713571703187f, 0.2673587081891581f},
    {0.9568627450980394f, 0.42745098039215673f, 0.26274509803921564f},
    {0.9520953479430986f, 0.42022299115724726f, 0.26459054209919286f},
    {0.9473279507881583f, 0.4129950019223377f, 0.26643598615916975f},
    {0.9425605536332179f, 0.40576701268742793f, 0.26828143021914663f},
    {0.9377931564782777f, 0.39853902345251835f, 0.2701268742791235f},
    {0.9330257593233372f, 0.3913110342176086f, 0.2719723183391004f},
    {0.928258362168397f, 0.38408304498269874f, 0.27381776239907707f},
    {0.9234909650134564f, 0.3768550557477892f, 0.2756632064590543f},
    {0.9187235678585162f, 0.36962706651287985f, 0.2775086505190312f},
    {0.9139561707035755f, 0.36239907727797027f, 0.2793540945790083f},
    {0.9091887735486351f, 0.3551710880430604f, 0.28119953863898506f},
    {0.9044213763936948f, 0.34794309880815055f, 0.28304498269896183f},
    {0.8996539792387543f, 0.340715109573241f, 0.28489042675893894f},
    {0.8948865820838141f, 0.3334871203383314f, 0.28673587081891583f},
    {0.8901191849288733f, 0.326259131103422f, 0.28858131487889305f},
    {0.8853517877739333f, 0.319031141868512f, 0.2904267589388695f},
    {0.8805843906189926f, 0.3118031526336025f, 0.2922722029988466f},
    {0.8758169934640523f, 0.30457516339869284f, 0.2941176470588236f},
    {0.871049596309112f, 0.2973471741637831f, 0.2959630911188005f},
    {0.8662821991541714f, 0.29011918492887356f, 0.2978085351787776f},
    {0.8615148019992313f, 0.2828911956939638f, 0.2996539792387542f},
    {0.8567474048442908f, 0.2756632064590542f, 0.30149942329873114f},
    {0.8519800076893502f, 0.26843521722414465f, 0.3033448673587078f},
    {0.84721261053441f, 0.2612072279892348f, 0.3051903114186846f},
    {0.8424452133794698f, 0.2539792387543254f, 0.3070357554786618f},
    {0.8376778162245291f, 0.24675124951941552f, 0.3088811995386387f},
    {0.8310649750096116f, 0.23844675124951953f, 0.30880430603613984f},
    {0.8226066897347173f, 0.22906574394463686f, 0.3068050749711647f},
    {0.8141484044598231f, 0.21968473663975407f, 0.30480584390618953f},
    {0.8056901191849288f, 0.2103037293348713f, 0.30280661284121513f},
    {0.7972318339100345f, 0.20092272202998862f, 0.30080738177624006f},
    {0.7887735486351404f, 0.19154171472510573f, 0.2988081507112643f},
    {0.7803152633602461f, 0.18216070742022294f, 0.2968089196462894f},
    {0.7718569780853518f, 0.17277970011534027f, 0.2948096885813144f},
    {0.7633986928104576f, 0.1633986928104575f, 0.29281045751633983f},
    {0.7549404075355632f, 0.15401768550557482f, 0.29081122645136503f},
    {0.746482122260669f, 0.14463667820069204f, 0.2888119953863893f},
    {0.7380238369857748f, 0.13525567089580925f, 0.2868127643214149f},
    {0.7295655517108804f, 0.12587466359092658f, 0.2848135332564399f},
    {0.7211072664359862f, 0.1164936562860438f, 0.28281430219146436f},
    {0.7126489811610921f, 0.10711264898116135f, 0.2808150711264894f},
    {0.7041906958861976f, 0.09773164167627835f, 0.2788158400615146f},
    {0.6957324106113034f, 0.08835063437139556f, 0.27681660899653987f},
    {0.6872741253364091f, 0.07896962706651289f, 0.2748173779315646f},
    {0.6788158400615149f, 0.06958861976163011f, 0.27281814686658995f},
    {0.6703575547866205f, 0.06020761245674744f, 0.2708189158016141f},
    {0.6618992695117263f, 0.05082660515186466f, 0.26881968473663936f},
    {0.6534409842368321f, 0.041445597846981874f, 0.2668204536716644f},
    {0.6449826989619377f, 0.0320645905420992f, 0.26482122260668983f},
    {0.6365244136870435f, 0.02268358323721642f, 0.2628219915417146f},
    {0.6280661284121491f, 0.013302575932333749f, 0.26082276047673913f}};

const size_t grey_n = 256;
const float grey_palette[grey_n][3] = {
    {0.2, 0.2, 0.2},
    {0.20392156862745098, 0.20392156862745098, 0.20392156862745098},
    {0.20392156862745098, 0.20392156862745098, 0.20392156862745098},
    {0.20784313725490194, 0.20784313725490194, 0.20784313725490194},
    {0.21176470588235294, 0.21176470588235294, 0.21176470588235294},
    {0.21568627450980393, 0.21568627450980393, 0.21568627450980393},
    {0.2196078431372549, 0.2196078431372549, 0.2196078431372549},
    {0.2196078431372549, 0.2196078431372549, 0.2196078431372549},
    {0.22352941176470587, 0.22352941176470587, 0.22352941176470587},
    {0.22745098039215686, 0.22745098039215686, 0.22745098039215686},
    {0.23137254901960785, 0.23137254901960785, 0.23137254901960785},
    {0.23529411764705882, 0.23529411764705882, 0.23529411764705882},
    {0.23529411764705882, 0.23529411764705882, 0.23529411764705882},
    {0.2392156862745098, 0.2392156862745098, 0.2392156862745098},
    {0.24313725490196078, 0.24313725490196078, 0.24313725490196078},
    {0.24705882352941178, 0.24705882352941178, 0.24705882352941178},
    {0.25098039215686274, 0.25098039215686274, 0.25098039215686274},
    {0.25098039215686274, 0.25098039215686274, 0.25098039215686274},
    {0.2549019607843137, 0.2549019607843137, 0.2549019607843137},
    {0.2588235294117647, 0.2588235294117647, 0.2588235294117647},
    {0.2627450980392157, 0.2627450980392157, 0.2627450980392157},
    {0.26666666666666666, 0.26666666666666666, 0.26666666666666666},
    {0.26666666666666666, 0.26666666666666666, 0.26666666666666666},
    {0.27058823529411763, 0.27058823529411763, 0.27058823529411763},
    {0.27450980392156865, 0.27450980392156865, 0.27450980392156865},
    {0.2784313725490196, 0.2784313725490196, 0.2784313725490196},
    {0.2823529411764706, 0.2823529411764706, 0.2823529411764706},
    {0.2823529411764706, 0.2823529411764706, 0.2823529411764706},
    {0.28627450980392155, 0.28627450980392155, 0.28627450980392155},
    {0.2901960784313725, 0.2901960784313725, 0.2901960784313725},
    {0.29411764705882354, 0.29411764705882354, 0.29411764705882354},
    {0.2980392156862745, 0.2980392156862745, 0.2980392156862745},
    {0.2980392156862745, 0.2980392156862745, 0.2980392156862745},
    {0.30196078431372547, 0.30196078431372547, 0.30196078431372547},
    {0.3058823529411765, 0.3058823529411765, 0.3058823529411765},
    {0.30980392156862746, 0.30980392156862746, 0.30980392156862746},
    {0.3137254901960784, 0.3137254901960784, 0.3137254901960784},
    {0.3137254901960784, 0.3137254901960784, 0.3137254901960784},
    {0.3176470588235294, 0.3176470588235294, 0.3176470588235294},
    {0.32156862745098036, 0.32156862745098036, 0.32156862745098036},
    {0.3254901960784314, 0.3254901960784314, 0.3254901960784314},
    {0.32941176470588235, 0.32941176470588235, 0.32941176470588235},
    {0.32941176470588235, 0.32941176470588235, 0.32941176470588235},
    {0.3333333333333333, 0.3333333333333333, 0.3333333333333333},
    {0.33725490196078434, 0.33725490196078434, 0.33725490196078434},
    {0.3411764705882353, 0.3411764705882353, 0.3411764705882353},
    {0.34509803921568627, 0.34509803921568627, 0.34509803921568627},
    {0.34509803921568627, 0.34509803921568627, 0.34509803921568627},
    {0.34901960784313724, 0.34901960784313724, 0.34901960784313724},
    {0.3529411764705882, 0.3529411764705882, 0.3529411764705882},
    {0.3568627450980392, 0.3568627450980392, 0.3568627450980392},
    {0.3607843137254902, 0.3607843137254902, 0.3607843137254902},
    {0.3607843137254902, 0.3607843137254902, 0.3607843137254902},
    {0.36470588235294116, 0.36470588235294116, 0.36470588235294116},
    {0.3686274509803922, 0.3686274509803922, 0.3686274509803922},
    {0.37254901960784315, 0.37254901960784315, 0.37254901960784315},
    {0.3764705882352941, 0.3764705882352941, 0.3764705882352941},
    {0.3764705882352941, 0.3764705882352941, 0.3764705882352941},
    {0.3803921568627451, 0.3803921568627451, 0.3803921568627451},
    {0.38431372549019605, 0.38431372549019605, 0.38431372549019605},
    {0.38823529411764707, 0.38823529411764707, 0.38823529411764707},
    {0.39215686274509803, 0.39215686274509803, 0.39215686274509803},
    {0.39215686274509803, 0.39215686274509803, 0.39215686274509803},
    {0.396078431372549, 0.396078431372549, 0.396078431372549},
    {0.4, 0.4, 0.4},
    {0.403921568627451, 0.403921568627451, 0.403921568627451},
    {0.40784313725490196, 0.40784313725490196, 0.40784313725490196},
    {0.40784313725490196, 0.40784313725490196, 0.40784313725490196},
    {0.4117647058823529, 0.4117647058823529, 0.4117647058823529},
    {0.4156862745098039, 0.4156862745098039, 0.4156862745098039},
    {0.4196078431372549, 0.4196078431372549, 0.4196078431372549},
    {0.4235294117647059, 0.4235294117647059, 0.4235294117647059},
    {0.4235294117647059, 0.4235294117647059, 0.4235294117647059},
    {0.42745098039215684, 0.42745098039215684, 0.42745098039215684},
    {0.43137254901960786, 0.43137254901960786, 0.43137254901960786},
    {0.43529411764705883, 0.43529411764705883, 0.43529411764705883},
    {0.4392156862745098, 0.4392156862745098, 0.4392156862745098},
    {0.4392156862745098, 0.4392156862745098, 0.4392156862745098},
    {0.44313725490196076, 0.44313725490196076, 0.44313725490196076},
    {0.44705882352941173, 0.44705882352941173, 0.44705882352941173},
    {0.45098039215686275, 0.45098039215686275, 0.45098039215686275},
    {0.4549019607843137, 0.4549019607843137, 0.4549019607843137},
    {0.4549019607843137, 0.4549019607843137, 0.4549019607843137},
    {0.4588235294117647, 0.4588235294117647, 0.4588235294117647},
    {0.4627450980392157, 0.4627450980392157, 0.4627450980392157},
    {0.4666666666666667, 0.4666666666666667, 0.4666666666666667},
    {0.47058823529411764, 0.47058823529411764, 0.47058823529411764},
    {0.47058823529411764, 0.47058823529411764, 0.47058823529411764},
    {0.4745098039215686, 0.4745098039215686, 0.4745098039215686},
    {0.4784313725490196, 0.4784313725490196, 0.4784313725490196},
    {0.4823529411764706, 0.4823529411764706, 0.4823529411764706},
    {0.48627450980392156, 0.48627450980392156, 0.48627450980392156},
    {0.48627450980392156, 0.48627450980392156, 0.48627450980392156},
    {0.49019607843137253, 0.49019607843137253, 0.49019607843137253},
    {0.49411764705882355, 0.49411764705882355, 0.49411764705882355},
    {0.4980392156862745, 0.4980392156862745, 0.4980392156862745},
    {0.5019607843137255, 0.5019607843137255, 0.5019607843137255},
    {0.5019607843137255, 0.5019607843137255, 0.5019607843137255},
    {0.5058823529411764, 0.5058823529411764, 0.5058823529411764},
    {0.5098039215686274, 0.5098039215686274, 0.5098039215686274},
    {0.5137254901960784, 0.5137254901960784, 0.5137254901960784},
    {0.5176470588235293, 0.5176470588235293, 0.5176470588235293},
    {0.5176470588235293, 0.5176470588235293, 0.5176470588235293},
    {0.5215686274509804, 0.5215686274509804, 0.5215686274509804},
    {0.5254901960784314, 0.5254901960784314, 0.5254901960784314},
    {0.5294117647058824, 0.5294117647058824, 0.5294117647058824},
    {0.5333333333333333, 0.5333333333333333, 0.5333333333333333},
    {0.5333333333333333, 0.5333333333333333, 0.5333333333333333},
    {0.5372549019607843, 0.5372549019607843, 0.5372549019607843},
    {0.5411764705882353, 0.5411764705882353, 0.5411764705882353},
    {0.5450980392156862, 0.5450980392156862, 0.5450980392156862},
    {0.5490196078431373, 0.5490196078431373, 0.5490196078431373},
    {0.5490196078431373, 0.5490196078431373, 0.5490196078431373},
    {0.5529411764705883, 0.5529411764705883, 0.5529411764705883},
    {0.5568627450980392, 0.5568627450980392, 0.5568627450980392},
    {0.5607843137254902, 0.5607843137254902, 0.5607843137254902},
    {0.5647058823529412, 0.5647058823529412, 0.5647058823529412},
    {0.5647058823529412, 0.5647058823529412, 0.5647058823529412},
    {0.5686274509803921, 0.5686274509803921, 0.5686274509803921},
    {0.5725490196078431, 0.5725490196078431, 0.5725490196078431},
    {0.5764705882352941, 0.5764705882352941, 0.5764705882352941},
    {0.580392156862745, 0.580392156862745, 0.580392156862745},
    {0.580392156862745, 0.580392156862745, 0.580392156862745},
    {0.5843137254901961, 0.5843137254901961, 0.5843137254901961},
    {0.5882352941176471, 0.5882352941176471, 0.5882352941176471},
    {0.592156862745098, 0.592156862745098, 0.592156862745098},
    {0.596078431372549, 0.596078431372549, 0.596078431372549},
    {0.596078431372549, 0.596078431372549, 0.596078431372549},
    {0.6, 0.6, 0.6},
    {0.6039215686274509, 0.6039215686274509, 0.6039215686274509},
    {0.6078431372549019, 0.6078431372549019, 0.6078431372549019},
    {0.611764705882353, 0.611764705882353, 0.611764705882353},
    {0.611764705882353, 0.611764705882353, 0.611764705882353},
    {0.615686274509804, 0.615686274509804, 0.615686274509804},
    {0.6196078431372549, 0.6196078431372549, 0.6196078431372549},
    {0.6235294117647059, 0.6235294117647059, 0.6235294117647059},
    {0.6274509803921569, 0.6274509803921569, 0.6274509803921569},
    {0.6274509803921569, 0.6274509803921569, 0.6274509803921569},
    {0.6313725490196078, 0.6313725490196078, 0.6313725490196078},
    {0.6352941176470588, 0.6352941176470588, 0.6352941176470588},
    {0.6392156862745098, 0.6392156862745098, 0.6392156862745098},
    {0.6431372549019607, 0.6431372549019607, 0.6431372549019607},
    {0.6431372549019607, 0.6431372549019607, 0.6431372549019607},
    {0.6470588235294118, 0.6470588235294118, 0.6470588235294118},
    {0.6509803921568628, 0.6509803921568628, 0.6509803921568628},
    {0.6549019607843137, 0.6549019607843137, 0.6549019607843137},
    {0.6588235294117647, 0.6588235294117647, 0.6588235294117647},
    {0.6588235294117647, 0.6588235294117647, 0.6588235294117647},
    {0.6627450980392157, 0.6627450980392157, 0.6627450980392157},
    {0.6666666666666666, 0.6666666666666666, 0.6666666666666666},
    {0.6705882352941176, 0.6705882352941176, 0.6705882352941176},
    {0.6745098039215687, 0.6745098039215687, 0.6745098039215687},
    {0.6745098039215687, 0.6745098039215687, 0.6745098039215687},
    {0.6784313725490196, 0.6784313725490196, 0.6784313725490196},
    {0.6823529411764706, 0.6823529411764706, 0.6823529411764706},
    {0.6862745098039216, 0.6862745098039216, 0.6862745098039216},
    {0.6901960784313725, 0.6901960784313725, 0.6901960784313725},
    {0.6901960784313725, 0.6901960784313725, 0.6901960784313725},
    {0.6941176470588235, 0.6941176470588235, 0.6941176470588235},
    {0.6980392156862745, 0.6980392156862745, 0.6980392156862745},
    {0.7019607843137254, 0.7019607843137254, 0.7019607843137254},
    {0.7058823529411764, 0.7058823529411764, 0.7058823529411764},
    {0.7058823529411764, 0.7058823529411764, 0.7058823529411764},
    {0.7098039215686275, 0.7098039215686275, 0.7098039215686275},
    {0.7137254901960784, 0.7137254901960784, 0.7137254901960784},
    {0.7176470588235294, 0.7176470588235294, 0.7176470588235294},
    {0.7215686274509804, 0.7215686274509804, 0.7215686274509804},
    {0.7215686274509804, 0.7215686274509804, 0.7215686274509804},
    {0.7254901960784313, 0.7254901960784313, 0.7254901960784313},
    {0.7294117647058823, 0.7294117647058823, 0.7294117647058823},
    {0.7333333333333333, 0.7333333333333333, 0.7333333333333333},
    {0.7372549019607844, 0.7372549019607844, 0.7372549019607844},
    {0.7372549019607844, 0.7372549019607844, 0.7372549019607844},
    {0.7411764705882353, 0.7411764705882353, 0.7411764705882353},
    {0.7450980392156863, 0.7450980392156863, 0.7450980392156863},
    {0.7490196078431373, 0.7490196078431373, 0.7490196078431373},
    {0.7529411764705882, 0.7529411764705882, 0.7529411764705882},
    {0.7529411764705882, 0.7529411764705882, 0.7529411764705882},
    {0.7568627450980392, 0.7568627450980392, 0.7568627450980392},
    {0.7607843137254902, 0.7607843137254902, 0.7607843137254902},
    {0.7647058823529411, 0.7647058823529411, 0.7647058823529411},
    {0.7686274509803921, 0.7686274509803921, 0.7686274509803921},
    {0.7686274509803921, 0.7686274509803921, 0.7686274509803921},
    {0.7725490196078432, 0.7725490196078432, 0.7725490196078432},
    {0.7764705882352941, 0.7764705882352941, 0.7764705882352941},
    {0.7803921568627451, 0.7803921568627451, 0.7803921568627451},
    {0.7843137254901961, 0.7843137254901961, 0.7843137254901961},
    {0.7843137254901961, 0.7843137254901961, 0.7843137254901961},
    {0.788235294117647, 0.788235294117647, 0.788235294117647},
    {0.792156862745098, 0.792156862745098, 0.792156862745098},
    {0.796078431372549, 0.796078431372549, 0.796078431372549},
    {0.8, 0.8, 0.8},
    {0.8, 0.8, 0.8},
    {0.803921568627451, 0.803921568627451, 0.803921568627451},
    {0.807843137254902, 0.807843137254902, 0.807843137254902},
    {0.8117647058823529, 0.8117647058823529, 0.8117647058823529},
    {0.8156862745098039, 0.8156862745098039, 0.8156862745098039},
    {0.8156862745098039, 0.8156862745098039, 0.8156862745098039},
    {0.8196078431372549, 0.8196078431372549, 0.8196078431372549},
    {0.8235294117647058, 0.8235294117647058, 0.8235294117647058},
    {0.8274509803921568, 0.8274509803921568, 0.8274509803921568},
    {0.8313725490196078, 0.8313725490196078, 0.8313725490196078},
    {0.8313725490196078, 0.8313725490196078, 0.8313725490196078},
    {0.8352941176470589, 0.8352941176470589, 0.8352941176470589},
    {0.8392156862745098, 0.8392156862745098, 0.8392156862745098},
    {0.8431372549019608, 0.8431372549019608, 0.8431372549019608},
    {0.8470588235294118, 0.8470588235294118, 0.8470588235294118},
    {0.8470588235294118, 0.8470588235294118, 0.8470588235294118},
    {0.8509803921568627, 0.8509803921568627, 0.8509803921568627},
    {0.8549019607843137, 0.8549019607843137, 0.8549019607843137},
    {0.8588235294117647, 0.8588235294117647, 0.8588235294117647},
    {0.8627450980392157, 0.8627450980392157, 0.8627450980392157},
    {0.8627450980392157, 0.8627450980392157, 0.8627450980392157},
    {0.8666666666666667, 0.8666666666666667, 0.8666666666666667},
    {0.8705882352941177, 0.8705882352941177, 0.8705882352941177},
    {0.8745098039215686, 0.8745098039215686, 0.8745098039215686},
    {0.8784313725490196, 0.8784313725490196, 0.8784313725490196},
    {0.8784313725490196, 0.8784313725490196, 0.8784313725490196},
    {0.8823529411764706, 0.8823529411764706, 0.8823529411764706},
    {0.8862745098039215, 0.8862745098039215, 0.8862745098039215},
    {0.8901960784313725, 0.8901960784313725, 0.8901960784313725},
    {0.8941176470588235, 0.8941176470588235, 0.8941176470588235},
    {0.8941176470588235, 0.8941176470588235, 0.8941176470588235},
    {0.8980392156862745, 0.8980392156862745, 0.8980392156862745},
    {0.9019607843137255, 0.9019607843137255, 0.9019607843137255},
    {0.9058823529411765, 0.9058823529411765, 0.9058823529411765},
    {0.9098039215686274, 0.9098039215686274, 0.9098039215686274},
    {0.9098039215686274, 0.9098039215686274, 0.9098039215686274},
    {0.9137254901960784, 0.9137254901960784, 0.9137254901960784},
    {0.9176470588235294, 0.9176470588235294, 0.9176470588235294},
    {0.9215686274509803, 0.9215686274509803, 0.9215686274509803},
    {0.9254901960784314, 0.9254901960784314, 0.9254901960784314},
    {0.9254901960784314, 0.9254901960784314, 0.9254901960784314},
    {0.9294117647058824, 0.9294117647058824, 0.9294117647058824},
    {0.9333333333333333, 0.9333333333333333, 0.9333333333333333},
    {0.9372549019607843, 0.9372549019607843, 0.9372549019607843},
    {0.9411764705882353, 0.9411764705882353, 0.9411764705882353},
    {0.9411764705882353, 0.9411764705882353, 0.9411764705882353},
    {0.9450980392156862, 0.9450980392156862, 0.9450980392156862},
    {0.9490196078431372, 0.9490196078431372, 0.9490196078431372},
    {0.9529411764705882, 0.9529411764705882, 0.9529411764705882},
    {0.9568627450980391, 0.9568627450980391, 0.9568627450980391},
    {0.9568627450980391, 0.9568627450980391, 0.9568627450980391},
    {0.9607843137254902, 0.9607843137254902, 0.9607843137254902},
    {0.9647058823529412, 0.9647058823529412, 0.9647058823529412},
    {0.9686274509803922, 0.9686274509803922, 0.9686274509803922},
    {0.9725490196078431, 0.9725490196078431, 0.9725490196078431},
    {0.9725490196078431, 0.9725490196078431, 0.9725490196078431},
    {0.9764705882352941, 0.9764705882352941, 0.9764705882352941},
    {0.9803921568627451, 0.9803921568627451, 0.9803921568627451},
    {0.984313725490196, 0.984313725490196, 0.984313725490196},
    {0.9882352941176471, 0.9882352941176471, 0.9882352941176471},
    {0.9882352941176471, 0.9882352941176471, 0.9882352941176471},
    {0.9921568627450981, 0.9921568627450981, 0.9921568627450981},
    {0.996078431372549, 0.996078431372549, 0.996078431372549},
    {1.0, 1.0, 1.0},
};
                                       
const size_t grey_cal_ref_n = 256;
const float grey_cal_ref_palette[grey_cal_ref_n][3] = {
    {0.2, 0.2, 0.2},
    {0.21176470588235294, 0.21176470588235294, 0.21176470588235294},
    {0.22745098039215686, 0.22745098039215686, 0.22745098039215686},
    {0.24313725490196078, 0.24313725490196078, 0.24313725490196078},
    {0.2588235294117647, 0.2588235294117647, 0.2588235294117647},
    {0.27450980392156865, 0.27450980392156865, 0.27450980392156865},
    {0.28627450980392155, 0.28627450980392155, 0.28627450980392155},
    {0.30196078431372547, 0.30196078431372547, 0.30196078431372547},
    {0.3137254901960784, 0.3137254901960784, 0.3137254901960784},
    {0.32941176470588235, 0.32941176470588235, 0.32941176470588235},
    {0.34509803921568627, 0.34509803921568627, 0.34509803921568627},
    {0.3568627450980392, 0.3568627450980392, 0.3568627450980392},
    {0.3686274509803922, 0.3686274509803922, 0.3686274509803922},
    {0.3803921568627451, 0.3803921568627451, 0.3803921568627451},
    {0.39215686274509803, 0.39215686274509803, 0.39215686274509803},
    {0.40784313725490196, 0.40784313725490196, 0.40784313725490196},
    {0.4196078431372549, 0.4196078431372549, 0.4196078431372549},
    {0.43137254901960786, 0.43137254901960786, 0.43137254901960786},
    {0.44313725490196076, 0.44313725490196076, 0.44313725490196076},
    {0.4549019607843137, 0.4549019607843137, 0.4549019607843137},
    {0.4666666666666667, 0.4666666666666667, 0.4666666666666667},
    {0.4784313725490196, 0.4784313725490196, 0.4784313725490196},
    {0.48627450980392156, 0.48627450980392156, 0.48627450980392156},
    {0.5019607843137255, 0.5019607843137255, 0.5019607843137255},
    {0.5098039215686274, 0.5098039215686274, 0.5098039215686274},
    {0.5215686274509804, 0.5215686274509804, 0.5215686274509804},
    {0.5333333333333333, 0.5333333333333333, 0.5333333333333333},
    {0.5411764705882353, 0.5411764705882353, 0.5411764705882353},
    {0.5490196078431373, 0.5490196078431373, 0.5490196078431373},
    {0.5647058823529412, 0.5647058823529412, 0.5647058823529412},
    {0.5725490196078431, 0.5725490196078431, 0.5725490196078431},
    {0.580392156862745, 0.580392156862745, 0.580392156862745},
    {0.592156862745098, 0.592156862745098, 0.592156862745098},
    {0.596078431372549, 0.596078431372549, 0.596078431372549},
    {0.6078431372549019, 0.6078431372549019, 0.6078431372549019},
    {0.615686274509804, 0.615686274509804, 0.615686274509804},
    {0.6274509803921569, 0.6274509803921569, 0.6274509803921569},
    {0.6352941176470588, 0.6352941176470588, 0.6352941176470588},
    {0.6431372549019607, 0.6431372549019607, 0.6431372549019607},
    {0.6509803921568628, 0.6509803921568628, 0.6509803921568628},
    {0.6588235294117647, 0.6588235294117647, 0.6588235294117647},
    {0.6666666666666666, 0.6666666666666666, 0.6666666666666666},
    {0.6745098039215687, 0.6745098039215687, 0.6745098039215687},
    {0.6823529411764706, 0.6823529411764706, 0.6823529411764706},
    {0.6901960784313725, 0.6901960784313725, 0.6901960784313725},
    {0.6980392156862745, 0.6980392156862745, 0.6980392156862745},
    {0.7058823529411764, 0.7058823529411764, 0.7058823529411764},
    {0.7098039215686275, 0.7098039215686275, 0.7098039215686275},
    {0.7176470588235294, 0.7176470588235294, 0.7176470588235294},
    {0.7215686274509804, 0.7215686274509804, 0.7215686274509804},
    {0.7333333333333333, 0.7333333333333333, 0.7333333333333333},
    {0.7372549019607844, 0.7372549019607844, 0.7372549019607844},
    {0.7450980392156863, 0.7450980392156863, 0.7450980392156863},
    {0.7529411764705882, 0.7529411764705882, 0.7529411764705882},
    {0.7568627450980392, 0.7568627450980392, 0.7568627450980392},
    {0.7647058823529411, 0.7647058823529411, 0.7647058823529411},
    {0.7686274509803921, 0.7686274509803921, 0.7686274509803921},
    {0.7725490196078432, 0.7725490196078432, 0.7725490196078432},
    {0.7803921568627451, 0.7803921568627451, 0.7803921568627451},
    {0.7843137254901961, 0.7843137254901961, 0.7843137254901961},
    {0.792156862745098, 0.792156862745098, 0.792156862745098},
    {0.796078431372549, 0.796078431372549, 0.796078431372549},
    {0.8, 0.8, 0.8},
    {0.807843137254902, 0.807843137254902, 0.807843137254902},
    {0.8117647058823529, 0.8117647058823529, 0.8117647058823529},
    {0.8156862745098039, 0.8156862745098039, 0.8156862745098039},
    {0.8196078431372549, 0.8196078431372549, 0.8196078431372549},
    {0.8274509803921568, 0.8274509803921568, 0.8274509803921568},
    {0.8313725490196078, 0.8313725490196078, 0.8313725490196078},
    {0.8352941176470589, 0.8352941176470589, 0.8352941176470589},
    {0.8392156862745098, 0.8392156862745098, 0.8392156862745098},
    {0.8431372549019608, 0.8431372549019608, 0.8431372549019608},
    {0.8470588235294118, 0.8470588235294118, 0.8470588235294118},
    {0.8509803921568627, 0.8509803921568627, 0.8509803921568627},
    {0.8549019607843137, 0.8549019607843137, 0.8549019607843137},
    {0.8627450980392157, 0.8627450980392157, 0.8627450980392157},
    {0.8627450980392157, 0.8627450980392157, 0.8627450980392157},
    {0.8666666666666667, 0.8666666666666667, 0.8666666666666667},
    {0.8705882352941177, 0.8705882352941177, 0.8705882352941177},
    {0.8745098039215686, 0.8745098039215686, 0.8745098039215686},
    {0.8784313725490196, 0.8784313725490196, 0.8784313725490196},
    {0.8823529411764706, 0.8823529411764706, 0.8823529411764706},
    {0.8862745098039215, 0.8862745098039215, 0.8862745098039215},
    {0.8901960784313725, 0.8901960784313725, 0.8901960784313725},
    {0.8941176470588235, 0.8941176470588235, 0.8941176470588235},
    {0.8941176470588235, 0.8941176470588235, 0.8941176470588235},
    {0.8980392156862745, 0.8980392156862745, 0.8980392156862745},
    {0.9019607843137255, 0.9019607843137255, 0.9019607843137255},
    {0.9058823529411765, 0.9058823529411765, 0.9058823529411765},
    {0.9098039215686274, 0.9098039215686274, 0.9098039215686274},
    {0.9098039215686274, 0.9098039215686274, 0.9098039215686274},
    {0.9098039215686274, 0.9098039215686274, 0.9098039215686274},
    {0.9137254901960784, 0.9137254901960784, 0.9137254901960784},
    {0.9176470588235294, 0.9176470588235294, 0.9176470588235294},
    {0.9215686274509803, 0.9215686274509803, 0.9215686274509803},
    {0.9254901960784314, 0.9254901960784314, 0.9254901960784314},
    {0.9254901960784314, 0.9254901960784314, 0.9254901960784314},
    {0.9254901960784314, 0.9254901960784314, 0.9254901960784314},
    {0.9294117647058824, 0.9294117647058824, 0.9294117647058824},
    {0.9333333333333333, 0.9333333333333333, 0.9333333333333333},
    {0.9333333333333333, 0.9333333333333333, 0.9333333333333333},
    {0.9372549019607843, 0.9372549019607843, 0.9372549019607843},
    {0.9411764705882353, 0.9411764705882353, 0.9411764705882353},
    {0.9411764705882353, 0.9411764705882353, 0.9411764705882353},
    {0.9411764705882353, 0.9411764705882353, 0.9411764705882353},
    {0.9411764705882353, 0.9411764705882353, 0.9411764705882353},
    {0.9450980392156862, 0.9450980392156862, 0.9450980392156862},
    {0.9490196078431372, 0.9490196078431372, 0.9490196078431372},
    {0.9490196078431372, 0.9490196078431372, 0.9490196078431372},
    {0.9529411764705882, 0.9529411764705882, 0.9529411764705882},
    {0.9529411764705882, 0.9529411764705882, 0.9529411764705882},
    {0.9568627450980391, 0.9568627450980391, 0.9568627450980391},
    {0.9568627450980391, 0.9568627450980391, 0.9568627450980391},
    {0.9568627450980391, 0.9568627450980391, 0.9568627450980391},
    {0.9568627450980391, 0.9568627450980391, 0.9568627450980391},
    {0.9607843137254902, 0.9607843137254902, 0.9607843137254902},
    {0.9607843137254902, 0.9607843137254902, 0.9607843137254902},
    {0.9647058823529412, 0.9647058823529412, 0.9647058823529412},
    {0.9647058823529412, 0.9647058823529412, 0.9647058823529412},
    {0.9647058823529412, 0.9647058823529412, 0.9647058823529412},
    {0.9686274509803922, 0.9686274509803922, 0.9686274509803922},
    {0.9686274509803922, 0.9686274509803922, 0.9686274509803922},
    {0.9725490196078431, 0.9725490196078431, 0.9725490196078431},
    {0.9725490196078431, 0.9725490196078431, 0.9725490196078431},
    {0.9725490196078431, 0.9725490196078431, 0.9725490196078431},
    {0.9725490196078431, 0.9725490196078431, 0.9725490196078431},
    {0.9725490196078431, 0.9725490196078431, 0.9725490196078431},
    {0.9725490196078431, 0.9725490196078431, 0.9725490196078431},
    {0.9764705882352941, 0.9764705882352941, 0.9764705882352941},
    {0.9764705882352941, 0.9764705882352941, 0.9764705882352941},
    {0.9764705882352941, 0.9764705882352941, 0.9764705882352941},
    {0.9803921568627451, 0.9803921568627451, 0.9803921568627451},
    {0.9803921568627451, 0.9803921568627451, 0.9803921568627451},
    {0.9803921568627451, 0.9803921568627451, 0.9803921568627451},
    {0.9803921568627451, 0.9803921568627451, 0.9803921568627451},
    {0.984313725490196, 0.984313725490196, 0.984313725490196},
    {0.984313725490196, 0.984313725490196, 0.984313725490196},
    {0.984313725490196, 0.984313725490196, 0.984313725490196},
    {0.984313725490196, 0.984313725490196, 0.984313725490196},
    {0.9882352941176471, 0.9882352941176471, 0.9882352941176471},
    {0.9882352941176471, 0.9882352941176471, 0.9882352941176471},
    {0.9882352941176471, 0.9882352941176471, 0.9882352941176471},
    {0.9882352941176471, 0.9882352941176471, 0.9882352941176471},
    {0.9882352941176471, 0.9882352941176471, 0.9882352941176471},
    {0.9882352941176471, 0.9882352941176471, 0.9882352941176471},
    {0.9882352941176471, 0.9882352941176471, 0.9882352941176471},
    {0.9882352941176471, 0.9882352941176471, 0.9882352941176471},
    {0.9882352941176471, 0.9882352941176471, 0.9882352941176471},
    {0.9882352941176471, 0.9882352941176471, 0.9882352941176471},
    {0.9882352941176471, 0.9882352941176471, 0.9882352941176471},
    {0.9882352941176471, 0.9882352941176471, 0.9882352941176471},
    {0.9921568627450981, 0.9921568627450981, 0.9921568627450981},
    {0.9921568627450981, 0.9921568627450981, 0.9921568627450981},
    {0.9921568627450981, 0.9921568627450981, 0.9921568627450981},
    {0.9921568627450981, 0.9921568627450981, 0.9921568627450981},
    {0.9921568627450981, 0.9921568627450981, 0.9921568627450981},
    {0.9921568627450981, 0.9921568627450981, 0.9921568627450981},
    {0.9921568627450981, 0.9921568627450981, 0.9921568627450981},
    {0.9921568627450981, 0.9921568627450981, 0.9921568627450981},
    {0.996078431372549, 0.996078431372549, 0.996078431372549},
    {0.996078431372549, 0.996078431372549, 0.996078431372549},
    {0.996078431372549, 0.996078431372549, 0.996078431372549},
    {0.996078431372549, 0.996078431372549, 0.996078431372549},
    {0.996078431372549, 0.996078431372549, 0.996078431372549},
    {0.996078431372549, 0.996078431372549, 0.996078431372549},
    {0.996078431372549, 0.996078431372549, 0.996078431372549},
    {0.996078431372549, 0.996078431372549, 0.996078431372549},
    {0.996078431372549, 0.996078431372549, 0.996078431372549},
    {0.996078431372549, 0.996078431372549, 0.996078431372549},
    {0.996078431372549, 0.996078431372549, 0.996078431372549},
    {0.996078431372549, 0.996078431372549, 0.996078431372549},
    {1.0, 1.0, 1.0},
    {1.0, 1.0, 1.0},
    {1.0, 1.0, 1.0},
    {1.0, 1.0, 1.0},
    {1.0, 1.0, 1.0},
    {1.0, 1.0, 1.0},
    {1.0, 1.0, 1.0},
    {1.0, 1.0, 1.0},
    {1.0, 1.0, 1.0},
    {1.0, 1.0, 1.0},
    {1.0, 1.0, 1.0},
    {1.0, 1.0, 1.0},
    {1.0, 1.0, 1.0},
    {1.0, 1.0, 1.0},
    {1.0, 1.0, 1.0},
    {1.0, 1.0, 1.0},
    {1.0, 1.0, 1.0},
    {1.0, 1.0, 1.0},
    {1.0, 1.0, 1.0},
    {1.0, 1.0, 1.0},
    {1.0, 1.0, 1.0},
    {1.0, 1.0, 1.0},
    {1.0, 1.0, 1.0},
    {1.0, 1.0, 1.0},
    {1.0, 1.0, 1.0},
    {1.0, 1.0, 1.0},
    {1.0, 1.0, 1.0},
    {1.0, 1.0, 1.0},
    {1.0, 1.0, 1.0},
    {1.0, 1.0, 1.0},
    {1.0, 1.0, 1.0},
    {1.0, 1.0, 1.0},
    {1.0, 1.0, 1.0},
    {1.0, 1.0, 1.0},
    {1.0, 1.0, 1.0},
    {1.0, 1.0, 1.0},
    {1.0, 1.0, 1.0},
    {1.0, 1.0, 1.0},
    {1.0, 1.0, 1.0},
    {1.0, 1.0, 1.0},
    {1.0, 1.0, 1.0},
    {1.0, 1.0, 1.0},
    {1.0, 1.0, 1.0},
    {1.0, 1.0, 1.0},
    {1.0, 1.0, 1.0},
    {1.0, 1.0, 1.0},
    {1.0, 1.0, 1.0},
    {1.0, 1.0, 1.0},
    {1.0, 1.0, 1.0},
    {1.0, 1.0, 1.0},
    {1.0, 1.0, 1.0},
    {1.0, 1.0, 1.0},
    {1.0, 1.0, 1.0},
    {1.0, 1.0, 1.0},
    {1.0, 1.0, 1.0},
    {1.0, 1.0, 1.0},
    {1.0, 1.0, 1.0},
    {1.0, 1.0, 1.0},
    {1.0, 1.0, 1.0},
    {1.0, 1.0, 1.0},
    {1.0, 1.0, 1.0},
    {1.0, 1.0, 1.0},
    {1.0, 1.0, 1.0},
    {1.0, 1.0, 1.0},
    {1.0, 1.0, 1.0},
    {1.0, 1.0, 1.0},
    {1.0, 1.0, 1.0},
    {1.0, 1.0, 1.0},
    {1.0, 1.0, 1.0},
    {1.0, 1.0, 1.0},
    {1.0, 1.0, 1.0},
    {1.0, 1.0, 1.0},
    {1.0, 1.0, 1.0},
    {1.0, 1.0, 1.0},
    {1.0, 1.0, 1.0},
    {1.0, 1.0, 1.0},
    {1.0, 1.0, 1.0},
    {1.0, 1.0, 1.0},
    {1.0, 1.0, 1.0},
    {1.0, 1.0, 1.0},
    {1.0, 1.0, 1.0},
    {1.0, 1.0, 1.0},
    {1.0, 1.0, 1.0},
    {1.0, 1.0, 1.0},
    {1.0, 1.0, 1.0},
};



// original colormaps licensed CC0f, public domain, no attribution needed:
// https://github.com/BIDS/colormap/blob/master/LICENSE.txt
const size_t viridis_n = 256;
const float viridis_palette[viridis_n][3] = {
    {0.26700401f, 0.00487433f, 0.32941519f},
    {0.26851048f, 0.00960483f, 0.33542652f},
    {0.26994384f, 0.01462494f, 0.34137895f},
    {0.27130489f, 0.01994186f, 0.34726862f},
    {0.27259384f, 0.02556309f, 0.35309303f},
    {0.27380934f, 0.03149748f, 0.35885256f},
    {0.27495242f, 0.03775181f, 0.36454323f},
    {0.27602238f, 0.04416723f, 0.37016418f},
    {0.2770184f, 0.05034437f, 0.37571452f},
    {0.27794143f, 0.05632444f, 0.38119074f},
    {0.27879067f, 0.06214536f, 0.38659204f},
    {0.2795655f, 0.06783587f, 0.39191723f},
    {0.28026658f, 0.07341724f, 0.39716349f},
    {0.28089358f, 0.07890703f, 0.40232944f},
    {0.28144581f, 0.0843197f, 0.40741404f},
    {0.28192358f, 0.08966622f, 0.41241521f},
    {0.28232739f, 0.09495545f, 0.41733086f},
    {0.28265633f, 0.10019576f, 0.42216032f},
    {0.28291049f, 0.10539345f, 0.42690202f},
    {0.28309095f, 0.11055307f, 0.43155375f},
    {0.28319704f, 0.11567966f, 0.43611482f},
    {0.28322882f, 0.12077701f, 0.44058404f},
    {0.28318684f, 0.12584799f, 0.44496f},
    {0.283072f, 0.13089477f, 0.44924127f},
    {0.28288389f, 0.13592005f, 0.45342734f},
    {0.28262297f, 0.14092556f, 0.45751726f},
    {0.28229037f, 0.14591233f, 0.46150995f},
    {0.28188676f, 0.15088147f, 0.46540474f},
    {0.28141228f, 0.15583425f, 0.46920128f},
    {0.28086773f, 0.16077132f, 0.47289909f},
    {0.28025468f, 0.16569272f, 0.47649762f},
    {0.27957399f, 0.17059884f, 0.47999675f},
    {0.27882618f, 0.1754902f, 0.48339654f},
    {0.27801236f, 0.18036684f, 0.48669702f},
    {0.27713437f, 0.18522836f, 0.48989831f},
    {0.27619376f, 0.19007447f, 0.49300074f},
    {0.27519116f, 0.1949054f, 0.49600488f},
    {0.27412802f, 0.19972086f, 0.49891131f},
    {0.27300596f, 0.20452049f, 0.50172076f},
    {0.27182812f, 0.20930306f, 0.50443413f},
    {0.27059473f, 0.21406899f, 0.50705243f},
    {0.26930756f, 0.21881782f, 0.50957678f},
    {0.26796846f, 0.22354911f, 0.5120084f},
    {0.26657984f, 0.2282621f, 0.5143487f},
    {0.2651445f, 0.23295593f, 0.5165993f},
    {0.2636632f, 0.23763078f, 0.51876163f},
    {0.26213801f, 0.24228619f, 0.52083736f},
    {0.26057103f, 0.2469217f, 0.52282822f},
    {0.25896451f, 0.25153685f, 0.52473609f},
    {0.25732244f, 0.2561304f, 0.52656332f},
    {0.25564519f, 0.26070284f, 0.52831152f},
    {0.25393498f, 0.26525384f, 0.52998273f},
    {0.25219404f, 0.26978306f, 0.53157905f},
    {0.25042462f, 0.27429024f, 0.53310261f},
    {0.24862899f, 0.27877509f, 0.53455561f},
    {0.2468114f, 0.28323662f, 0.53594093f},
    {0.24497208f, 0.28767547f, 0.53726018f},
    {0.24311324f, 0.29209154f, 0.53851561f},
    {0.24123708f, 0.29648471f, 0.53970946f},
    {0.23934575f, 0.30085494f, 0.54084398f},
    {0.23744138f, 0.30520222f, 0.5419214f},
    {0.23552606f, 0.30952657f, 0.54294396f},
    {0.23360277f, 0.31382773f, 0.54391424f},
    {0.2316735f, 0.3181058f, 0.54483444f},
    {0.22973926f, 0.32236127f, 0.54570633f},
    {0.22780192f, 0.32659432f, 0.546532f},
    {0.2258633f, 0.33080515f, 0.54731353f},
    {0.22392515f, 0.334994f, 0.54805291f},
    {0.22198915f, 0.33916114f, 0.54875211f},
    {0.22005691f, 0.34330688f, 0.54941304f},
    {0.21812995f, 0.34743154f, 0.55003755f},
    {0.21620971f, 0.35153548f, 0.55062743f},
    {0.21429757f, 0.35561907f, 0.5511844f},
    {0.21239477f, 0.35968273f, 0.55171011f},
    {0.2105031f, 0.36372671f, 0.55220646f},
    {0.20862342f, 0.36775151f, 0.55267486f},
    {0.20675628f, 0.37175775f, 0.55311653f},
    {0.20490257f, 0.37574589f, 0.55353282f},
    {0.20306309f, 0.37971644f, 0.55392505f},
    {0.20123854f, 0.38366989f, 0.55429441f},
    {0.1994295f, 0.38760678f, 0.55464205f},
    {0.1976365f, 0.39152762f, 0.55496905f},
    {0.19585993f, 0.39543297f, 0.55527637f},
    {0.19410009f, 0.39932336f, 0.55556494f},
    {0.19235719f, 0.40319934f, 0.55583559f},
    {0.19063135f, 0.40706148f, 0.55608907f},
    {0.18892259f, 0.41091033f, 0.55632606f},
    {0.18723083f, 0.41474645f, 0.55654717f},
    {0.18555593f, 0.4185704f, 0.55675292f},
    {0.18389763f, 0.42238275f, 0.55694377f},
    {0.18225561f, 0.42618405f, 0.5571201f},
    {0.18062949f, 0.42997486f, 0.55728221f},
    {0.17901879f, 0.43375572f, 0.55743035f},
    {0.17742298f, 0.4375272f, 0.55756466f},
    {0.17584148f, 0.44128981f, 0.55768526f},
    {0.17427363f, 0.4450441f, 0.55779216f},
    {0.17271876f, 0.4487906f, 0.55788532f},
    {0.17117615f, 0.4525298f, 0.55796464f},
    {0.16964573f, 0.45626209f, 0.55803034f},
    {0.16812641f, 0.45998802f, 0.55808199f},
    {0.1666171f, 0.46370813f, 0.55811913f},
    {0.16511703f, 0.4674229f, 0.55814141f},
    {0.16362543f, 0.47113278f, 0.55814842f},
    {0.16214155f, 0.47483821f, 0.55813967f},
    {0.16066467f, 0.47853961f, 0.55811466f},
    {0.15919413f, 0.4822374f, 0.5580728f},
    {0.15772933f, 0.48593197f, 0.55801347f},
    {0.15626973f, 0.4896237f, 0.557936f},
    {0.15481488f, 0.49331293f, 0.55783967f},
    {0.15336445f, 0.49700003f, 0.55772371f},
    {0.1519182f, 0.50068529f, 0.55758733f},
    {0.15047605f, 0.50436904f, 0.55742968f},
    {0.14903918f, 0.50805136f, 0.5572505f},
    {0.14760731f, 0.51173263f, 0.55704861f},
    {0.14618026f, 0.51541316f, 0.55682271f},
    {0.14475863f, 0.51909319f, 0.55657181f},
    {0.14334327f, 0.52277292f, 0.55629491f},
    {0.14193527f, 0.52645254f, 0.55599097f},
    {0.14053599f, 0.53013219f, 0.55565893f},
    {0.13914708f, 0.53381201f, 0.55529773f},
    {0.13777048f, 0.53749213f, 0.55490625f},
    {0.1364085f, 0.54117264f, 0.55448339f},
    {0.13506561f, 0.54485335f, 0.55402906f},
    {0.13374299f, 0.54853458f, 0.55354108f},
    {0.13244401f, 0.55221637f, 0.55301828f},
    {0.13117249f, 0.55589872f, 0.55245948f},
    {0.1299327f, 0.55958162f, 0.55186354f},
    {0.12872938f, 0.56326503f, 0.55122927f},
    {0.12756771f, 0.56694891f, 0.55055551f},
    {0.12645338f, 0.57063316f, 0.5498411f},
    {0.12539383f, 0.57431754f, 0.54908564f},
    {0.12439474f, 0.57800205f, 0.5482874f},
    {0.12346281f, 0.58168661f, 0.54744498f},
    {0.12260562f, 0.58537105f, 0.54655722f},
    {0.12183122f, 0.58905521f, 0.54562298f},
    {0.12114807f, 0.59273889f, 0.54464114f},
    {0.12056501f, 0.59642187f, 0.54361058f},
    {0.12009154f, 0.60010387f, 0.54253043f},
    {0.11973756f, 0.60378459f, 0.54139999f},
    {0.11951163f, 0.60746388f, 0.54021751f},
    {0.11942341f, 0.61114146f, 0.53898192f},
    {0.11948255f, 0.61481702f, 0.53769219f},
    {0.11969858f, 0.61849025f, 0.53634733f},
    {0.12008079f, 0.62216081f, 0.53494633f},
    {0.12063824f, 0.62582833f, 0.53348834f},
    {0.12137972f, 0.62949242f, 0.53197275f},
    {0.12231244f, 0.63315277f, 0.53039808f},
    {0.12344358f, 0.63680899f, 0.52876343f},
    {0.12477953f, 0.64046069f, 0.52706792f},
    {0.12632581f, 0.64410744f, 0.52531069f},
    {0.12808703f, 0.64774881f, 0.52349092f},
    {0.13006688f, 0.65138436f, 0.52160791f},
    {0.13226797f, 0.65501363f, 0.51966086f},
    {0.13469183f, 0.65863619f, 0.5176488f},
    {0.13733921f, 0.66225157f, 0.51557101f},
    {0.14020991f, 0.66585927f, 0.5134268f},
    {0.14330291f, 0.66945881f, 0.51121549f},
    {0.1466164f, 0.67304968f, 0.50893644f},
    {0.15014782f, 0.67663139f, 0.5065889f},
    {0.15389405f, 0.68020343f, 0.50417217f},
    {0.15785146f, 0.68376525f, 0.50168574f},
    {0.16201598f, 0.68731632f, 0.49912906f},
    {0.1663832f, 0.69085611f, 0.49650163f},
    {0.1709484f, 0.69438405f, 0.49380294f},
    {0.17570671f, 0.6978996f, 0.49103252f},
    {0.18065314f, 0.70140222f, 0.48818938f},
    {0.18578266f, 0.70489133f, 0.48527326f},
    {0.19109018f, 0.70836635f, 0.48228395f},
    {0.19657063f, 0.71182668f, 0.47922108f},
    {0.20221902f, 0.71527175f, 0.47608431f},
    {0.20803045f, 0.71870095f, 0.4728733f},
    {0.21400015f, 0.72211371f, 0.46958774f},
    {0.22012381f, 0.72550945f, 0.46622638f},
    {0.2263969f, 0.72888753f, 0.46278934f},
    {0.23281498f, 0.73224735f, 0.45927675f},
    {0.2393739f, 0.73558828f, 0.45568838f},
    {0.24606968f, 0.73890972f, 0.45202405f},
    {0.25289851f, 0.74221104f, 0.44828355f},
    {0.25985676f, 0.74549162f, 0.44446673f},
    {0.26694127f, 0.74875084f, 0.44057284f},
    {0.27414922f, 0.75198807f, 0.4366009f},
    {0.28147681f, 0.75520266f, 0.43255207f},
    {0.28892102f, 0.75839399f, 0.42842626f},
    {0.29647899f, 0.76156142f, 0.42422341f},
    {0.30414796f, 0.76470433f, 0.41994346f},
    {0.31192534f, 0.76782207f, 0.41558638f},
    {0.3198086f, 0.77091403f, 0.41115215f},
    {0.3277958f, 0.77397953f, 0.40664011f},
    {0.33588539f, 0.7770179f, 0.40204917f},
    {0.34407411f, 0.78002855f, 0.39738103f},
    {0.35235985f, 0.78301086f, 0.39263579f},
    {0.36074053f, 0.78596419f, 0.38781353f},
    {0.3692142f, 0.78888793f, 0.38291438f},
    {0.37777892f, 0.79178146f, 0.3779385f},
    {0.38643282f, 0.79464415f, 0.37288606f},
    {0.39517408f, 0.79747541f, 0.36775726f},
    {0.40400101f, 0.80027461f, 0.36255223f},
    {0.4129135f, 0.80304099f, 0.35726893f},
    {0.42190813f, 0.80577412f, 0.35191009f},
    {0.43098317f, 0.80847343f, 0.34647607f},
    {0.44013691f, 0.81113836f, 0.3409673f},
    {0.44936763f, 0.81376835f, 0.33538426f},
    {0.45867362f, 0.81636288f, 0.32972749f},
    {0.46805314f, 0.81892143f, 0.32399761f},
    {0.47750446f, 0.82144351f, 0.31819529f},
    {0.4870258f, 0.82392862f, 0.31232133f},
    {0.49661536f, 0.82637633f, 0.30637661f},
    {0.5062713f, 0.82878621f, 0.30036211f},
    {0.51599182f, 0.83115784f, 0.29427888f},
    {0.52577622f, 0.83349064f, 0.2881265f},
    {0.5356211f, 0.83578452f, 0.28190832f},
    {0.5455244f, 0.83803918f, 0.27562602f},
    {0.55548397f, 0.84025437f, 0.26928147f},
    {0.5654976f, 0.8424299f, 0.26287683f},
    {0.57556297f, 0.84456561f, 0.25641457f},
    {0.58567772f, 0.84666139f, 0.24989748f},
    {0.59583934f, 0.84871722f, 0.24332878f},
    {0.60604528f, 0.8507331f, 0.23671214f},
    {0.61629283f, 0.85270912f, 0.23005179f},
    {0.62657923f, 0.85464543f, 0.22335258f},
    {0.63690157f, 0.85654226f, 0.21662012f},
    {0.64725685f, 0.85839991f, 0.20986086f},
    {0.65764197f, 0.86021878f, 0.20308229f},
    {0.66805369f, 0.86199932f, 0.19629307f},
    {0.67848868f, 0.86374211f, 0.18950326f},
    {0.68894351f, 0.86544779f, 0.18272455f},
    {0.69941463f, 0.86711711f, 0.17597055f},
    {0.70989842f, 0.86875092f, 0.16925712f},
    {0.72039115f, 0.87035015f, 0.16260273f},
    {0.73088902f, 0.87191584f, 0.15602894f},
    {0.74138803f, 0.87344918f, 0.14956101f},
    {0.75188414f, 0.87495143f, 0.14322828f},
    {0.76237342f, 0.87642392f, 0.13706449f},
    {0.77285183f, 0.87786808f, 0.13110864f},
    {0.78331535f, 0.87928545f, 0.12540538f},
    {0.79375994f, 0.88067763f, 0.12000532f},
    {0.80418159f, 0.88204632f, 0.11496505f},
    {0.81457634f, 0.88339329f, 0.11034678f},
    {0.82494028f, 0.88472036f, 0.10621724f},
    {0.83526959f, 0.88602943f, 0.1026459f},
    {0.84556056f, 0.88732243f, 0.09970219f},
    {0.8558096f, 0.88860134f, 0.09745186f},
    {0.86601325f, 0.88986815f, 0.09595277f},
    {0.87616824f, 0.89112487f, 0.09525046f},
    {0.88627146f, 0.89237353f, 0.09537439f},
    {0.89632002f, 0.89361614f, 0.09633538f},
    {0.90631121f, 0.89485467f, 0.09812496f},
    {0.91624212f, 0.89609127f, 0.1007168f},
    {0.92610579f, 0.89732977f, 0.10407067f},
    {0.93590444f, 0.8985704f, 0.10813094f},
    {0.94563626f, 0.899815f, 0.11283773f},
    {0.95529972f, 0.90106534f, 0.11812832f},
    {0.96489353f, 0.90232311f, 0.12394051f},
    {0.97441665f, 0.90358991f, 0.13021494f},
    {0.98386829f, 0.90486726f, 0.13689671f},
    {0.99324789f, 0.90615657f, 0.1439362f}};
    
const size_t viridis_cal_ref_n = 256;
const float viridis_cal_ref_palette[viridis_cal_ref_n][3] = {
    {0.26700401, 0.00487433, 0.32941519},
    {0.27259384, 0.02556309, 0.35309303},
    {0.27794143, 0.05632444, 0.38119074},
    {0.28144581, 0.0843197, 0.40741404},
    {0.28309095, 0.11055307, 0.43155375},
    {0.28288389, 0.13592005, 0.45342734},
    {0.28141228, 0.15583425, 0.46920128},
    {0.27801236, 0.18036684, 0.48669702},
    {0.27412802, 0.19972086, 0.49891131},
    {0.26796846, 0.22354911, 0.5120084},
    {0.26213801, 0.24228619, 0.52083736},
    {0.25564519, 0.26070284, 0.52831152},
    {0.24862899, 0.27877509, 0.53455561},
    {0.24123708, 0.29648471, 0.53970946},
    {0.23360277, 0.31382773, 0.54391424},
    {0.2258633, 0.33080515, 0.54731353},
    {0.21812995, 0.34743154, 0.55003755},
    {0.2105031, 0.36372671, 0.55220646},
    {0.20306309, 0.37971644, 0.55392505},
    {0.19585993, 0.39543297, 0.55527637},
    {0.19063135, 0.40706148, 0.55608907},
    {0.18389763, 0.42238275, 0.55694377},
    {0.17901879, 0.43375572, 0.55743035},
    {0.17271876, 0.4487906, 0.55788532},
    {0.16812641, 0.45998802, 0.55808199},
    {0.16214155, 0.47483821, 0.55813967},
    {0.15772933, 0.48593197, 0.55801347},
    {0.15336445, 0.49700003, 0.55772371},
    {0.14903918, 0.50805136, 0.5572505},
    {0.14334327, 0.52277292, 0.55629491},
    {0.13914708, 0.53381201, 0.55529773},
    {0.13506561, 0.54485335, 0.55402906},
    {0.13117249, 0.55589872, 0.55245948},
    {0.12872938, 0.56326503, 0.55122927},
    {0.12539383, 0.57431754, 0.54908564},
    {0.12260562, 0.58537105, 0.54655722},
    {0.12056501, 0.59642187, 0.54361058},
    {0.11951163, 0.60746388, 0.54021751},
    {0.11948255, 0.61481702, 0.53769219},
    {0.12063824, 0.62582833, 0.53348834},
    {0.12231244, 0.63315277, 0.53039808},
    {0.12632581, 0.64410744, 0.52531069},
    {0.13006688, 0.65138436, 0.52160791},
    {0.13733921, 0.66225157, 0.51557101},
    {0.14330291, 0.66945881, 0.51121549},
    {0.15389405, 0.68020343, 0.50417217},
    {0.16201598, 0.68731632, 0.49912906},
    {0.1709484, 0.69438405, 0.49380294},
    {0.18065314, 0.70140222, 0.48818938},
    {0.19109018, 0.70836635, 0.48228395},
    {0.20803045, 0.71870095, 0.4728733},
    {0.22012381, 0.72550945, 0.46622638},
    {0.23281498, 0.73224735, 0.45927675},
    {0.24606968, 0.73890972, 0.45202405},
    {0.25985676, 0.74549162, 0.44446673},
    {0.27414922, 0.75198807, 0.4366009},
    {0.28147681, 0.75520266, 0.43255207},
    {0.29647899, 0.76156142, 0.42422341},
    {0.31192534, 0.76782207, 0.41558638},
    {0.3277958, 0.77397953, 0.40664011},
    {0.34407411, 0.78002855, 0.39738103},
    {0.35235985, 0.78301086, 0.39263579},
    {0.3692142, 0.78888793, 0.38291438},
    {0.38643282, 0.79464415, 0.37288606},
    {0.39517408, 0.79747541, 0.36775726},
    {0.4129135, 0.80304099, 0.35726893},
    {0.42190813, 0.80577412, 0.35191009},
    {0.44013691, 0.81113836, 0.3409673},
    {0.44936763, 0.81376835, 0.33538426},
    {0.46805314, 0.81892143, 0.32399761},
    {0.47750446, 0.82144351, 0.31819529},
    {0.4870258, 0.82392862, 0.31232133},
    {0.5062713, 0.82878621, 0.30036211},
    {0.51599182, 0.83115784, 0.29427888},
    {0.52577622, 0.83349064, 0.2881265},
    {0.5455244, 0.83803918, 0.27562602},
    {0.55548397, 0.84025437, 0.26928147},
    {0.5654976, 0.8424299, 0.26287683},
    {0.57556297, 0.84456561, 0.25641457},
    {0.58567772, 0.84666139, 0.24989748},
    {0.60604528, 0.8507331, 0.23671214},
    {0.61629283, 0.85270912, 0.23005179},
    {0.62657923, 0.85464543, 0.22335258},
    {0.63690157, 0.85654226, 0.21662012},
    {0.64725685, 0.85839991, 0.20986086},
    {0.65764197, 0.86021878, 0.20308229},
    {0.66805369, 0.86199932, 0.19629307},
    {0.67848868, 0.86374211, 0.18950326},
    {0.68894351, 0.86544779, 0.18272455},
    {0.69941463, 0.86711711, 0.17597055},
    {0.69941463, 0.86711711, 0.17597055},
    {0.70989842, 0.86875092, 0.16925712},
    {0.72039115, 0.87035015, 0.16260273},
    {0.73088902, 0.87191584, 0.15602894},
    {0.74138803, 0.87344918, 0.14956101},
    {0.75188414, 0.87495143, 0.14322828},
    {0.75188414, 0.87495143, 0.14322828},
    {0.76237342, 0.87642392, 0.13706449},
    {0.77285183, 0.87786808, 0.13110864},
    {0.78331535, 0.87928545, 0.12540538},
    {0.78331535, 0.87928545, 0.12540538},
    {0.79375994, 0.88067763, 0.12000532},
    {0.80418159, 0.88204632, 0.11496505},
    {0.80418159, 0.88204632, 0.11496505},
    {0.81457634, 0.88339329, 0.11034678},
    {0.81457634, 0.88339329, 0.11034678},
    {0.82494028, 0.88472036, 0.10621724},
    {0.83526959, 0.88602943, 0.1026459},
    {0.83526959, 0.88602943, 0.1026459},
    {0.84556056, 0.88732243, 0.09970219},
    {0.84556056, 0.88732243, 0.09970219},
    {0.8558096, 0.88860134, 0.09745186},
    {0.8558096, 0.88860134, 0.09745186},
    {0.86601325, 0.88986815, 0.09595277},
    {0.86601325, 0.88986815, 0.09595277},
    {0.87616824, 0.89112487, 0.09525046},
    {0.87616824, 0.89112487, 0.09525046},
    {0.88627146, 0.89237353, 0.09537439},
    {0.88627146, 0.89237353, 0.09537439},
    {0.88627146, 0.89237353, 0.09537439},
    {0.89632002, 0.89361614, 0.09633538},
    {0.89632002, 0.89361614, 0.09633538},
    {0.90631121, 0.89485467, 0.09812496},
    {0.90631121, 0.89485467, 0.09812496},
    {0.90631121, 0.89485467, 0.09812496},
    {0.91624212, 0.89609127, 0.1007168},
    {0.91624212, 0.89609127, 0.1007168},
    {0.91624212, 0.89609127, 0.1007168},
    {0.92610579, 0.89732977, 0.10407067},
    {0.92610579, 0.89732977, 0.10407067},
    {0.92610579, 0.89732977, 0.10407067},
    {0.93590444, 0.8985704, 0.10813094},
    {0.93590444, 0.8985704, 0.10813094},
    {0.93590444, 0.8985704, 0.10813094},
    {0.93590444, 0.8985704, 0.10813094},
    {0.94563626, 0.899815, 0.11283773},
    {0.94563626, 0.899815, 0.11283773},
    {0.94563626, 0.899815, 0.11283773},
    {0.94563626, 0.899815, 0.11283773},
    {0.95529972, 0.90106534, 0.11812832},
    {0.95529972, 0.90106534, 0.11812832},
    {0.95529972, 0.90106534, 0.11812832},
    {0.95529972, 0.90106534, 0.11812832},
    {0.95529972, 0.90106534, 0.11812832},
    {0.95529972, 0.90106534, 0.11812832},
    {0.96489353, 0.90232311, 0.12394051},
    {0.96489353, 0.90232311, 0.12394051},
    {0.96489353, 0.90232311, 0.12394051},
    {0.96489353, 0.90232311, 0.12394051},
    {0.96489353, 0.90232311, 0.12394051},
    {0.96489353, 0.90232311, 0.12394051},
    {0.97441665, 0.90358991, 0.13021494},
    {0.97441665, 0.90358991, 0.13021494},
    {0.97441665, 0.90358991, 0.13021494},
    {0.97441665, 0.90358991, 0.13021494},
    {0.97441665, 0.90358991, 0.13021494},
    {0.97441665, 0.90358991, 0.13021494},
    {0.97441665, 0.90358991, 0.13021494},
    {0.97441665, 0.90358991, 0.13021494},
    {0.98386829, 0.90486726, 0.13689671},
    {0.98386829, 0.90486726, 0.13689671},
    {0.98386829, 0.90486726, 0.13689671},
    {0.98386829, 0.90486726, 0.13689671},
    {0.98386829, 0.90486726, 0.13689671},
    {0.98386829, 0.90486726, 0.13689671},
    {0.98386829, 0.90486726, 0.13689671},
    {0.98386829, 0.90486726, 0.13689671},
    {0.98386829, 0.90486726, 0.13689671},
    {0.98386829, 0.90486726, 0.13689671},
    {0.98386829, 0.90486726, 0.13689671},
    {0.98386829, 0.90486726, 0.13689671},
    {0.99324789, 0.90615657, 0.1439362},
    {0.99324789, 0.90615657, 0.1439362},
    {0.99324789, 0.90615657, 0.1439362},
    {0.99324789, 0.90615657, 0.1439362},
    {0.99324789, 0.90615657, 0.1439362},
    {0.99324789, 0.90615657, 0.1439362},
    {0.99324789, 0.90615657, 0.1439362},
    {0.99324789, 0.90615657, 0.1439362},
    {0.99324789, 0.90615657, 0.1439362},
    {0.99324789, 0.90615657, 0.1439362},
    {0.99324789, 0.90615657, 0.1439362},
    {0.99324789, 0.90615657, 0.1439362},
    {0.99324789, 0.90615657, 0.1439362},
    {0.99324789, 0.90615657, 0.1439362},
    {0.99324789, 0.90615657, 0.1439362},
    {0.99324789, 0.90615657, 0.1439362},
    {0.99324789, 0.90615657, 0.1439362},
    {0.99324789, 0.90615657, 0.1439362},
    {0.99324789, 0.90615657, 0.1439362},
    {0.99324789, 0.90615657, 0.1439362},
    {0.99324789, 0.90615657, 0.1439362},
    {0.99324789, 0.90615657, 0.1439362},
    {0.99324789, 0.90615657, 0.1439362},
    {0.99324789, 0.90615657, 0.1439362},
    {0.99324789, 0.90615657, 0.1439362},
    {0.99324789, 0.90615657, 0.1439362},
    {0.99324789, 0.90615657, 0.1439362},
    {0.99324789, 0.90615657, 0.1439362},
    {0.99324789, 0.90615657, 0.1439362},
    {0.99324789, 0.90615657, 0.1439362},
    {0.99324789, 0.90615657, 0.1439362},
    {0.99324789, 0.90615657, 0.1439362},
    {0.99324789, 0.90615657, 0.1439362},
    {0.99324789, 0.90615657, 0.1439362},
    {0.99324789, 0.90615657, 0.1439362},
    {0.99324789, 0.90615657, 0.1439362},
    {0.99324789, 0.90615657, 0.1439362},
    {0.99324789, 0.90615657, 0.1439362},
    {0.99324789, 0.90615657, 0.1439362},
    {0.99324789, 0.90615657, 0.1439362},
    {0.99324789, 0.90615657, 0.1439362},
    {0.99324789, 0.90615657, 0.1439362},
    {0.99324789, 0.90615657, 0.1439362},
    {0.99324789, 0.90615657, 0.1439362},
    {0.99324789, 0.90615657, 0.1439362},
    {0.99324789, 0.90615657, 0.1439362},
    {0.99324789, 0.90615657, 0.1439362},
    {0.99324789, 0.90615657, 0.1439362},
    {0.99324789, 0.90615657, 0.1439362},
    {0.99324789, 0.90615657, 0.1439362},
    {0.99324789, 0.90615657, 0.1439362},
    {0.99324789, 0.90615657, 0.1439362},
    {0.99324789, 0.90615657, 0.1439362},
    {0.99324789, 0.90615657, 0.1439362},
    {0.99324789, 0.90615657, 0.1439362},
    {0.99324789, 0.90615657, 0.1439362},
    {0.99324789, 0.90615657, 0.1439362},
    {0.99324789, 0.90615657, 0.1439362},
    {0.99324789, 0.90615657, 0.1439362},
    {0.99324789, 0.90615657, 0.1439362},
    {0.99324789, 0.90615657, 0.1439362},
    {0.99324789, 0.90615657, 0.1439362},
    {0.99324789, 0.90615657, 0.1439362},
    {0.99324789, 0.90615657, 0.1439362},
    {0.99324789, 0.90615657, 0.1439362},
    {0.99324789, 0.90615657, 0.1439362},
    {0.99324789, 0.90615657, 0.1439362},
    {0.99324789, 0.90615657, 0.1439362},
    {0.99324789, 0.90615657, 0.1439362},
    {0.99324789, 0.90615657, 0.1439362},
    {0.99324789, 0.90615657, 0.1439362},
    {0.99324789, 0.90615657, 0.1439362},
    {0.99324789, 0.90615657, 0.1439362},
    {0.99324789, 0.90615657, 0.1439362},
    {0.99324789, 0.90615657, 0.1439362},
    {0.99324789, 0.90615657, 0.1439362},
    {0.99324789, 0.90615657, 0.1439362},
    {0.99324789, 0.90615657, 0.1439362},
    {0.99324789, 0.90615657, 0.1439362},
    {0.99324789, 0.90615657, 0.1439362},
    {0.99324789, 0.90615657, 0.1439362},
    {0.99324789, 0.90615657, 0.1439362},
    {0.99324789, 0.90615657, 0.1439362},
    {0.99324789, 0.90615657, 0.1439362},
    {0.99324789, 0.90615657, 0.1439362},
};


const size_t magma_n = 256;
const float magma_palette[magma_n][3] = {
    {0.232077, 0.059889, 0.437695},
    {0.238826, 0.059517, 0.443256},
    {0.238826, 0.059517, 0.443256},
    {0.245543, 0.059352, 0.448436},
    {0.25222, 0.059415, 0.453248},
    {0.258857, 0.059706, 0.45771},
    {0.265447, 0.060237, 0.46184},
    {0.265447, 0.060237, 0.46184},
    {0.271994, 0.060994, 0.46566},
    {0.278493, 0.061978, 0.46919},
    {0.284951, 0.063168, 0.472451},
    {0.291366, 0.064553, 0.475462},
    {0.291366, 0.064553, 0.475462},
    {0.29774, 0.066117, 0.478243},
    {0.304081, 0.067835, 0.480812},
    {0.310382, 0.069702, 0.483186},
    {0.316654, 0.07169, 0.48538},
    {0.316654, 0.07169, 0.48538},
    {0.322899, 0.073782, 0.487408},
    {0.329114, 0.075972, 0.489287},
    {0.335308, 0.078236, 0.491024},
    {0.341482, 0.080564, 0.492631},
    {0.341482, 0.080564, 0.492631},
    {0.347636, 0.082946, 0.494121},
    {0.353773, 0.085373, 0.495501},
    {0.359898, 0.087831, 0.496778},
    {0.366012, 0.090314, 0.49796},
    {0.366012, 0.090314, 0.49796},
    {0.372116, 0.092816, 0.499053},
    {0.378211, 0.095332, 0.500067},
    {0.384299, 0.097855, 0.501002},
    {0.390384, 0.100379, 0.501864},
    {0.390384, 0.100379, 0.501864},
    {0.396467, 0.102902, 0.502658},
    {0.402548, 0.10542, 0.503386},
    {0.408629, 0.10793, 0.504052},
    {0.414709, 0.110431, 0.504662},
    {0.414709, 0.110431, 0.504662},
    {0.420791, 0.11292, 0.505215},
    {0.426877, 0.115395, 0.505714},
    {0.432967, 0.117855, 0.50616},
    {0.439062, 0.120298, 0.506555},
    {0.439062, 0.120298, 0.506555},
    {0.445163, 0.122724, 0.506901},
    {0.451271, 0.125132, 0.507198},
    {0.457386, 0.127522, 0.507448},
    {0.463508, 0.129893, 0.507652},
    {0.463508, 0.129893, 0.507652},
    {0.46964, 0.132245, 0.507809},
    {0.47578, 0.134577, 0.507921},
    {0.481929, 0.136891, 0.507989},
    {0.488088, 0.139186, 0.508011},
    {0.488088, 0.139186, 0.508011},
    {0.494258, 0.141462, 0.507988},
    {0.500438, 0.143719, 0.50792},
    {0.506629, 0.145958, 0.507806},
    {0.512831, 0.148179, 0.507648},
    {0.512831, 0.148179, 0.507648},
    {0.519045, 0.150383, 0.507443},
    {0.52527, 0.152569, 0.507192},
    {0.531507, 0.154739, 0.506895},
    {0.537755, 0.156894, 0.506551},
    {0.537755, 0.156894, 0.506551},
    {0.544015, 0.159033, 0.506159},
    {0.550287, 0.161158, 0.505719},
    {0.556571, 0.163269, 0.50523},
    {0.562866, 0.165368, 0.504692},
    {0.562866, 0.165368, 0.504692},
    {0.569172, 0.167454, 0.504105},
    {0.57549, 0.16953, 0.503466},
    {0.581819, 0.171596, 0.502777},
    {0.588158, 0.173652, 0.502035},
    {0.588158, 0.173652, 0.502035},
    {0.594508, 0.175701, 0.501241},
    {0.600868, 0.177743, 0.500394},
    {0.607238, 0.179779, 0.499492},
    {0.613617, 0.181811, 0.498536},
    {0.613617, 0.181811, 0.498536},
    {0.620005, 0.18384, 0.497524},
    {0.626401, 0.185867, 0.496456},
    {0.632805, 0.187893, 0.495332},
    {0.639216, 0.189921, 0.49415},
    {0.639216, 0.189921, 0.49415},
    {0.645633, 0.191952, 0.49291},
    {0.652056, 0.193986, 0.491611},
    {0.658483, 0.196027, 0.490253},
    {0.664915, 0.198075, 0.488836},
    {0.664915, 0.198075, 0.488836},
    {0.671349, 0.200133, 0.487358},
    {0.677786, 0.202203, 0.485819},
    {0.684224, 0.204286, 0.484219},
    {0.690661, 0.206384, 0.482558},
    {0.690661, 0.206384, 0.482558},
    {0.697098, 0.208501, 0.480835},
    {0.703532, 0.210638, 0.479049},
    {0.709962, 0.212797, 0.477201},
    {0.716387, 0.214982, 0.47529},
    {0.716387, 0.214982, 0.47529},
    {0.722805, 0.217194, 0.473316},
    {0.729216, 0.219437, 0.471279},
    {0.735616, 0.221713, 0.46918},
    {0.742004, 0.224025, 0.467018},
    {0.742004, 0.224025, 0.467018},
    {0.748378, 0.226377, 0.464794},
    {0.754737, 0.228772, 0.462509},
    {0.761077, 0.231214, 0.460162},
    {0.767398, 0.233705, 0.457755},
    {0.767398, 0.233705, 0.457755},
    {0.773695, 0.236249, 0.455289},
    {0.779968, 0.238851, 0.452765},
    {0.786212, 0.241514, 0.450184},
    {0.792427, 0.244242, 0.447543},
    {0.792427, 0.244242, 0.447543},
    {0.798608, 0.24704, 0.444848},
    {0.804752, 0.249911, 0.442102},
    {0.810855, 0.252861, 0.439305},
    {0.816914, 0.255895, 0.436461},
    {0.816914, 0.255895, 0.436461},
    {0.822926, 0.259016, 0.433573},
    {0.828886, 0.262229, 0.430644},
    {0.834791, 0.26554, 0.427671},
    {0.840636, 0.268953, 0.424666},
    {0.840636, 0.268953, 0.424666},
    {0.846416, 0.272473, 0.421631},
    {0.852126, 0.276106, 0.418573},
    {0.857763, 0.279857, 0.415496},
    {0.86332, 0.283729, 0.412403},
    {0.86332, 0.283729, 0.412403},
    {0.868793, 0.287728, 0.409303},
    {0.874176, 0.291859, 0.406205},
    {0.879464, 0.296125, 0.403118},
    {0.884651, 0.30053, 0.400047},
    {0.884651, 0.30053, 0.400047},
    {0.889731, 0.305079, 0.397002},
    {0.8947, 0.309773, 0.393995},
    {0.899552, 0.314616, 0.391037},
    {0.904281, 0.31961, 0.388137},
    {0.904281, 0.31961, 0.388137},
    {0.908884, 0.324755, 0.385308},
    {0.913354, 0.330052, 0.382563},
    {0.917689, 0.3355, 0.379915},
    {0.921884, 0.341098, 0.377376},
    {0.921884, 0.341098, 0.377376},
    {0.925937, 0.346844, 0.374959},
    {0.929845, 0.352734, 0.372677},
    {0.933606, 0.358764, 0.370541},
    {0.937221, 0.364929, 0.368567},
    {0.937221, 0.364929, 0.368567},
    {0.940687, 0.371224, 0.366762},
    {0.944006, 0.377643, 0.365136},
    {0.94718, 0.384178, 0.363701},
    {0.95021, 0.39082, 0.362468},
    {0.95021, 0.39082, 0.362468},
    {0.953099, 0.397563, 0.361438},
    {0.955849, 0.4044, 0.360619},
    {0.958464, 0.411324, 0.360014},
    {0.960949, 0.418323, 0.35963},
    {0.960949, 0.418323, 0.35963},
    {0.96331, 0.42539, 0.359469},
    {0.965549, 0.432519, 0.359529},
    {0.967671, 0.439703, 0.35981},
    {0.96968, 0.446936, 0.360311},
    {0.96968, 0.446936, 0.360311},
    {0.971582, 0.45421, 0.36103},
    {0.973381, 0.46152, 0.361965},
    {0.975082, 0.468861, 0.363111},
    {0.97669, 0.476226, 0.364466},
    {0.97669, 0.476226, 0.364466},
    {0.97821, 0.483612, 0.366025},
    {0.979645, 0.491014, 0.367783},
    {0.981, 0.498428, 0.369734},
    {0.982279, 0.505851, 0.371874},
    {0.982279, 0.505851, 0.371874},
    {0.983485, 0.51328, 0.374198},
    {0.984622, 0.520713, 0.376698},
    {0.985693, 0.528148, 0.379371},
    {0.9867, 0.535582, 0.38221},
    {0.9867, 0.535582, 0.38221},
    {0.987646, 0.543015, 0.38521},
    {0.988533, 0.550446, 0.388365},
    {0.989363, 0.557873, 0.391671},
    {0.990138, 0.565296, 0.395122},
    {0.990138, 0.565296, 0.395122},
    {0.990871, 0.572706, 0.398714},
    {0.991558, 0.580107, 0.402441},
    {0.992196, 0.587502, 0.406299},
    {0.992785, 0.594891, 0.410283},
    {0.992785, 0.594891, 0.410283},
    {0.993326, 0.602275, 0.41439},
    {0.993834, 0.609644, 0.418613},
    {0.994309, 0.616999, 0.42295},
    {0.994738, 0.62435, 0.427397},
    {0.994738, 0.62435, 0.427397},
    {0.995122, 0.631696, 0.431951},
    {0.99548, 0.639027, 0.436607},
    {0.99581, 0.646344, 0.441361},
    {0.996096, 0.653659, 0.446213},
    {0.996096, 0.653659, 0.446213},
    {0.996341, 0.660969, 0.45116},
    {0.99658, 0.668256, 0.456192},
    {0.996775, 0.675541, 0.461314},
    {0.996925, 0.682828, 0.466526},
    {0.996925, 0.682828, 0.466526},
    {0.997077, 0.690088, 0.471811},
    {0.997186, 0.697349, 0.477182},
    {0.997254, 0.704611, 0.482635},
    {0.997325, 0.711848, 0.488154},
    {0.997325, 0.711848, 0.488154},
    {0.997351, 0.719089, 0.493755},
    {0.997351, 0.726324, 0.499428},
    {0.997341, 0.733545, 0.505167},
    {0.997285, 0.740772, 0.510983},
    {0.997285, 0.740772, 0.510983},
    {0.997228, 0.747981, 0.516859},
    {0.997138, 0.75519, 0.522806},
    {0.997019, 0.762398, 0.528821},
    {0.996898, 0.769591, 0.534892},
    {0.996898, 0.769591, 0.534892},
    {0.996727, 0.776795, 0.541039},
    {0.996571, 0.783977, 0.547233},
    {0.996369, 0.791167, 0.553499},
    {0.996162, 0.798348, 0.55982},
    {0.996162, 0.798348, 0.55982},
    {0.995932, 0.805527, 0.566202},
    {0.99568, 0.812706, 0.572645},
    {0.995424, 0.819875, 0.57914},
    {0.995131, 0.827052, 0.585701},
    {0.995131, 0.827052, 0.585701},
    {0.994851, 0.834213, 0.592307},
    {0.994524, 0.841387, 0.598983},
    {0.994222, 0.84854, 0.605696},
    {0.993866, 0.855711, 0.612482},
    {0.993866, 0.855711, 0.612482},
    {0.993545, 0.862859, 0.619299},
    {0.99317, 0.870024, 0.626189},
    {0.992831, 0.877168, 0.633109},
    {0.99244, 0.88433, 0.640099},
    {0.99244, 0.88433, 0.640099},
    {0.992089, 0.89147, 0.647116},
    {0.991688, 0.898627, 0.654202},
    {0.991332, 0.905763, 0.661309},
    {0.99093, 0.912915, 0.668481},
    {0.99093, 0.912915, 0.668481},
    {0.99057, 0.920049, 0.675675},
    {0.990175, 0.927196, 0.682926},
    {0.989815, 0.934329, 0.690198},
    {0.989434, 0.94147, 0.697519},
    {0.989434, 0.94147, 0.697519},
    {0.989077, 0.948604, 0.704863},
    {0.988717, 0.955742, 0.712242},
    {0.988367, 0.962878, 0.719649},
    {0.988033, 0.970012, 0.727077},
    {0.988033, 0.970012, 0.727077},
    {0.987691, 0.977154, 0.734536},
    {0.987387, 0.984288, 0.742002},
    {0.987053, 0.991438, 0.749504},
};
    
const size_t magma_cal_ref_n = 256;
const float magma_cal_ref_palette[magma_cal_ref_n][3] = {
    {0.232077, 0.059889, 0.437695},
    {0.25222, 0.059415, 0.453248},
    {0.278493, 0.061978, 0.46919},
    {0.304081, 0.067835, 0.480812},
    {0.329114, 0.075972, 0.489287},
    {0.353773, 0.085373, 0.495501},
    {0.372116, 0.092816, 0.499053},
    {0.396467, 0.102902, 0.502658},
    {0.414709, 0.110431, 0.504662},
    {0.439062, 0.120298, 0.506555},
    {0.463508, 0.129893, 0.507652},
    {0.481929, 0.136891, 0.507989},
    {0.500438, 0.143719, 0.50792},
    {0.519045, 0.150383, 0.507443},
    {0.537755, 0.156894, 0.506551},
    {0.562866, 0.165368, 0.504692},
    {0.581819, 0.171596, 0.502777},
    {0.600868, 0.177743, 0.500394},
    {0.620005, 0.18384, 0.497524},
    {0.639216, 0.189921, 0.49415},
    {0.658483, 0.196027, 0.490253},
    {0.677786, 0.202203, 0.485819},
    {0.690661, 0.206384, 0.482558},
    {0.716387, 0.214982, 0.47529},
    {0.729216, 0.219437, 0.471279},
    {0.748378, 0.226377, 0.464794},
    {0.767398, 0.233705, 0.457755},
    {0.779968, 0.238851, 0.452765},
    {0.792427, 0.244242, 0.447543},
    {0.816914, 0.255895, 0.436461},
    {0.828886, 0.262229, 0.430644},
    {0.840636, 0.268953, 0.424666},
    {0.857763, 0.279857, 0.415496},
    {0.86332, 0.283729, 0.412403},
    {0.879464, 0.296125, 0.403118},
    {0.889731, 0.305079, 0.397002},
    {0.904281, 0.31961, 0.388137},
    {0.913354, 0.330052, 0.382563},
    {0.921884, 0.341098, 0.377376},
    {0.929845, 0.352734, 0.372677},
    {0.937221, 0.364929, 0.368567},
    {0.944006, 0.377643, 0.365136},
    {0.95021, 0.39082, 0.362468},
    {0.955849, 0.4044, 0.360619},
    {0.960949, 0.418323, 0.35963},
    {0.965549, 0.432519, 0.359529},
    {0.96968, 0.446936, 0.360311},
    {0.971582, 0.45421, 0.36103},
    {0.975082, 0.468861, 0.363111},
    {0.97669, 0.476226, 0.364466},
    {0.981, 0.498428, 0.369734},
    {0.982279, 0.505851, 0.371874},
    {0.984622, 0.520713, 0.376698},
    {0.9867, 0.535582, 0.38221},
    {0.987646, 0.543015, 0.38521},
    {0.989363, 0.557873, 0.391671},
    {0.990138, 0.565296, 0.395122},
    {0.990871, 0.572706, 0.398714},
    {0.992196, 0.587502, 0.406299},
    {0.992785, 0.594891, 0.410283},
    {0.993834, 0.609644, 0.418613},
    {0.994309, 0.616999, 0.42295},
    {0.994738, 0.62435, 0.427397},
    {0.99548, 0.639027, 0.436607},
    {0.99581, 0.646344, 0.441361},
    {0.996096, 0.653659, 0.446213},
    {0.996341, 0.660969, 0.45116},
    {0.996775, 0.675541, 0.461314},
    {0.996925, 0.682828, 0.466526},
    {0.997077, 0.690088, 0.471811},
    {0.997186, 0.697349, 0.477182},
    {0.997254, 0.704611, 0.482635},
    {0.997325, 0.711848, 0.488154},
    {0.997351, 0.719089, 0.493755},
    {0.997351, 0.726324, 0.499428},
    {0.997285, 0.740772, 0.510983},
    {0.997285, 0.740772, 0.510983},
    {0.997228, 0.747981, 0.516859},
    {0.997138, 0.75519, 0.522806},
    {0.997019, 0.762398, 0.528821},
    {0.996898, 0.769591, 0.534892},
    {0.996727, 0.776795, 0.541039},
    {0.996571, 0.783977, 0.547233},
    {0.996369, 0.791167, 0.553499},
    {0.996162, 0.798348, 0.55982},
    {0.996162, 0.798348, 0.55982},
    {0.995932, 0.805527, 0.566202},
    {0.99568, 0.812706, 0.572645},
    {0.995424, 0.819875, 0.57914},
    {0.995131, 0.827052, 0.585701},
    {0.995131, 0.827052, 0.585701},
    {0.995131, 0.827052, 0.585701},
    {0.994851, 0.834213, 0.592307},
    {0.994524, 0.841387, 0.598983},
    {0.994222, 0.84854, 0.605696},
    {0.993866, 0.855711, 0.612482},
    {0.993866, 0.855711, 0.612482},
    {0.993866, 0.855711, 0.612482},
    {0.993545, 0.862859, 0.619299},
    {0.99317, 0.870024, 0.626189},
    {0.99317, 0.870024, 0.626189},
    {0.992831, 0.877168, 0.633109},
    {0.99244, 0.88433, 0.640099},
    {0.99244, 0.88433, 0.640099},
    {0.99244, 0.88433, 0.640099},
    {0.99244, 0.88433, 0.640099},
    {0.992089, 0.89147, 0.647116},
    {0.991688, 0.898627, 0.654202},
    {0.991688, 0.898627, 0.654202},
    {0.991332, 0.905763, 0.661309},
    {0.991332, 0.905763, 0.661309},
    {0.99093, 0.912915, 0.668481},
    {0.99093, 0.912915, 0.668481},
    {0.99093, 0.912915, 0.668481},
    {0.99093, 0.912915, 0.668481},
    {0.99057, 0.920049, 0.675675},
    {0.99057, 0.920049, 0.675675},
    {0.990175, 0.927196, 0.682926},
    {0.990175, 0.927196, 0.682926},
    {0.990175, 0.927196, 0.682926},
    {0.989815, 0.934329, 0.690198},
    {0.989815, 0.934329, 0.690198},
    {0.989434, 0.94147, 0.697519},
    {0.989434, 0.94147, 0.697519},
    {0.989434, 0.94147, 0.697519},
    {0.989434, 0.94147, 0.697519},
    {0.989434, 0.94147, 0.697519},
    {0.989434, 0.94147, 0.697519},
    {0.989077, 0.948604, 0.704863},
    {0.989077, 0.948604, 0.704863},
    {0.989077, 0.948604, 0.704863},
    {0.988717, 0.955742, 0.712242},
    {0.988717, 0.955742, 0.712242},
    {0.988717, 0.955742, 0.712242},
    {0.988717, 0.955742, 0.712242},
    {0.988367, 0.962878, 0.719649},
    {0.988367, 0.962878, 0.719649},
    {0.988367, 0.962878, 0.719649},
    {0.988367, 0.962878, 0.719649},
    {0.988033, 0.970012, 0.727077},
    {0.988033, 0.970012, 0.727077},
    {0.988033, 0.970012, 0.727077},
    {0.988033, 0.970012, 0.727077},
    {0.988033, 0.970012, 0.727077},
    {0.988033, 0.970012, 0.727077},
    {0.988033, 0.970012, 0.727077},
    {0.988033, 0.970012, 0.727077},
    {0.988033, 0.970012, 0.727077},
    {0.988033, 0.970012, 0.727077},
    {0.988033, 0.970012, 0.727077},
    {0.988033, 0.970012, 0.727077},
    {0.987691, 0.977154, 0.734536},
    {0.987691, 0.977154, 0.734536},
    {0.987691, 0.977154, 0.734536},
    {0.987691, 0.977154, 0.734536},
    {0.987691, 0.977154, 0.734536},
    {0.987691, 0.977154, 0.734536},
    {0.987691, 0.977154, 0.734536},
    {0.987691, 0.977154, 0.734536},
    {0.987387, 0.984288, 0.742002},
    {0.987387, 0.984288, 0.742002},
    {0.987387, 0.984288, 0.742002},
    {0.987387, 0.984288, 0.742002},
    {0.987387, 0.984288, 0.742002},
    {0.987387, 0.984288, 0.742002},
    {0.987387, 0.984288, 0.742002},
    {0.987387, 0.984288, 0.742002},
    {0.987387, 0.984288, 0.742002},
    {0.987387, 0.984288, 0.742002},
    {0.987387, 0.984288, 0.742002},
    {0.987387, 0.984288, 0.742002},
    {0.987053, 0.991438, 0.749504},
    {0.987053, 0.991438, 0.749504},
    {0.987053, 0.991438, 0.749504},
    {0.987053, 0.991438, 0.749504},
    {0.987053, 0.991438, 0.749504},
    {0.987053, 0.991438, 0.749504},
    {0.987053, 0.991438, 0.749504},
    {0.987053, 0.991438, 0.749504},
    {0.987053, 0.991438, 0.749504},
    {0.987053, 0.991438, 0.749504},
    {0.987053, 0.991438, 0.749504},
    {0.987053, 0.991438, 0.749504},
    {0.987053, 0.991438, 0.749504},
    {0.987053, 0.991438, 0.749504},
    {0.987053, 0.991438, 0.749504},
    {0.987053, 0.991438, 0.749504},
    {0.987053, 0.991438, 0.749504},
    {0.987053, 0.991438, 0.749504},
    {0.987053, 0.991438, 0.749504},
    {0.987053, 0.991438, 0.749504},
    {0.987053, 0.991438, 0.749504},
    {0.987053, 0.991438, 0.749504},
    {0.987053, 0.991438, 0.749504},
    {0.987053, 0.991438, 0.749504},
    {0.987053, 0.991438, 0.749504},
    {0.987053, 0.991438, 0.749504},
    {0.987053, 0.991438, 0.749504},
    {0.987053, 0.991438, 0.749504},
    {0.987053, 0.991438, 0.749504},
    {0.987053, 0.991438, 0.749504},
    {0.987053, 0.991438, 0.749504},
    {0.987053, 0.991438, 0.749504},
    {0.987053, 0.991438, 0.749504},
    {0.987053, 0.991438, 0.749504},
    {0.987053, 0.991438, 0.749504},
    {0.987053, 0.991438, 0.749504},
    {0.987053, 0.991438, 0.749504},
    {0.987053, 0.991438, 0.749504},
    {0.987053, 0.991438, 0.749504},
    {0.987053, 0.991438, 0.749504},
    {0.987053, 0.991438, 0.749504},
    {0.987053, 0.991438, 0.749504},
    {0.987053, 0.991438, 0.749504},
    {0.987053, 0.991438, 0.749504},
    {0.987053, 0.991438, 0.749504},
    {0.987053, 0.991438, 0.749504},
    {0.987053, 0.991438, 0.749504},
    {0.987053, 0.991438, 0.749504},
    {0.987053, 0.991438, 0.749504},
    {0.987053, 0.991438, 0.749504},
    {0.987053, 0.991438, 0.749504},
    {0.987053, 0.991438, 0.749504},
    {0.987053, 0.991438, 0.749504},
    {0.987053, 0.991438, 0.749504},
    {0.987053, 0.991438, 0.749504},
    {0.987053, 0.991438, 0.749504},
    {0.987053, 0.991438, 0.749504},
    {0.987053, 0.991438, 0.749504},
    {0.987053, 0.991438, 0.749504},
    {0.987053, 0.991438, 0.749504},
    {0.987053, 0.991438, 0.749504},
    {0.987053, 0.991438, 0.749504},
    {0.987053, 0.991438, 0.749504},
    {0.987053, 0.991438, 0.749504},
    {0.987053, 0.991438, 0.749504},
    {0.987053, 0.991438, 0.749504},
    {0.987053, 0.991438, 0.749504},
    {0.987053, 0.991438, 0.749504},
    {0.987053, 0.991438, 0.749504},
    {0.987053, 0.991438, 0.749504},
    {0.987053, 0.991438, 0.749504},
    {0.987053, 0.991438, 0.749504},
    {0.987053, 0.991438, 0.749504},
    {0.987053, 0.991438, 0.749504},
    {0.987053, 0.991438, 0.749504},
    {0.987053, 0.991438, 0.749504},
    {0.987053, 0.991438, 0.749504},
    {0.987053, 0.991438, 0.749504},
    {0.987053, 0.991438, 0.749504},
    {0.987053, 0.991438, 0.749504},
    {0.987053, 0.991438, 0.749504},
    {0.987053, 0.991438, 0.749504},
    {0.987053, 0.991438, 0.749504},
    {0.987053, 0.991438, 0.749504},
    {0.987053, 0.991438, 0.749504},
};

// clang-format on
}  // namespace viz
}  // namespace ouster
